package ru.yandex.chemodan.app.djfs.core.history;

import java.util.Objects;

import lombok.RequiredArgsConstructor;
import org.bson.types.ObjectId;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.EventManager;
import ru.yandex.chemodan.app.djfs.core.album.Album;
import ru.yandex.chemodan.app.djfs.core.album.AlbumItem;
import ru.yandex.chemodan.app.djfs.core.filesystem.AlbumAppendActivity;
import ru.yandex.chemodan.app.djfs.core.filesystem.AlbumItemRemoveActivity;
import ru.yandex.chemodan.app.djfs.core.filesystem.AlbumRemoveActivity;
import ru.yandex.chemodan.app.djfs.core.filesystem.AlbumSetAttrActivity;
import ru.yandex.chemodan.app.djfs.core.filesystem.CopyActivity;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.MoveActivity;
import ru.yandex.chemodan.app.djfs.core.filesystem.event.PrivateFolderCreatedEvent;
import ru.yandex.chemodan.app.djfs.core.filesystem.event.SharedFolderCreatedEvent;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FilesystemPermission;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.MediaType;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfo;
import ru.yandex.chemodan.app.djfs.core.share.event.UserKickedFromGroupEvent;
import ru.yandex.chemodan.app.djfs.core.share.event.UserLeftGroupEvent;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.chemodan.util.tskv.TskvUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author eoshch
 */
@RequiredArgsConstructor
public class EventHistoryLogger {
    private static final Logger logger = LoggerFactory.getLogger("event-history");

    private final DjfsResourceDao djfsResourceDao;

    @EventManager.EventHandler
    public void handle(PrivateFolderCreatedEvent event) {
        if (!event.folder.getPath().getArea().isEventHistoryLogged()) {
            return;
        }

        FolderDjfsResource folder = event.folder;

        MapF<String, String> data = Cf.hashMap();
        data.put("event_type", "fs-mkdir");
        data.put("tgt_folder_id",
                folder.getResourceId().map(DjfsResourceId::toString).getOrElse(folder.getUid().asString() + ":"));
        data.put("tgt_rawaddress", folder.getPath().getFullPath());
        data.put("uid", folder.getUid().asString());
        data.put("resource_file_id",
                folder.getResourceId().map(DjfsResourceId::getFileId).map(DjfsFileId::getValue).getOrElse(""));
        data.put("owner_uid", folder.getUid().asString());
        data.put("resource_type", "dir");

        log(data);
    }

    @EventManager.EventHandler
    public void handle(SharedFolderCreatedEvent event) {
        if (!event.folder.getPath().getArea().isEventHistoryLogged()) {
            return;
        }

        FolderDjfsResource folder = event.folder;
        ShareInfo shareInfo = event.shareInfo;

        for (DjfsUid uid : shareInfo.allUids()) {
            MapF<String, String> data = Cf.hashMap();
            data.put("event_type", "fs-mkdir");
            data.put("tgt_folder_id",
                    folder.getResourceId().map(DjfsResourceId::toString).getOrElse(folder.getUid().asString() + ":"));
            data.put("user_uid", event.actor.asString());
            data.put("tgt_rawaddress", shareInfo.ownerPathToParticipantPath(folder.getPath(), uid).get().getFullPath());
            data.put("uid", uid.asString());
            data.put("owner_uid", folder.getUid().asString());
            data.put("resource_file_id",
                    folder.getResourceId().map(DjfsResourceId::getFileId).map(DjfsFileId::getValue).getOrElse(""));
            data.put("resource_type", "dir");

            log(data);
        }
    }

    @EventManager.EventHandler
    public void handle(UserLeftGroupEvent event) {
        for (DjfsUid uid : event.getShareInfo().allUids()) {
            MapF<String, String> data = Cf.hashMap();
            data.put("event_type", "share-leave-group");
            data.put("gid", event.getShareInfo().getGroupId());
            data.put("owner_uid", event.getShareInfo().getOwnerUid().asString());
            data.put("user_uid", event.getUid().asString());
            data.put("uid", uid.asString());
            data.put("path", event.getShareInfo().getRootPath(uid).get().getPath());
            data.put("rights", Integer.toString(event.getShareInfo().getPermissions(uid).get().value()));
            // count of remaining participants
            data.put("subscriber_count", Integer.toString(event.getShareInfo().allUids().length() - 2));
            data.put("is_invite", "False");
            log(data);
        }
    }

    @EventManager.EventHandler
    public void handle(UserKickedFromGroupEvent event) {
        for (DjfsUid uid : event.getShareInfo().allUids()) {
            MapF<String, String> data = Cf.hashMap();
            data.put("event_type", "share-kick-from-group");
            data.put("gid", event.getShareInfo().getGroupId());
            data.put("owner_uid", event.getShareInfo().getOwnerUid().asString());
            data.put("user_uid", event.getUid().asString());
            data.put("uid", uid.asString());
            data.put("path", event.getShareInfo().getRootPath(uid).get().getPath());
            data.put("rights", Integer.toString(event.getShareInfo().getPermissions(uid).get().value()));
            // count of remaining participants
            data.put("subscriber_count", Integer.toString(event.getShareInfo().allUids().length() - 2));
            data.put("is_invite", "False");
            log(data);
        }
    }

    private void formatAlbumLog(Album album, AlbumItem albumItem, DjfsUid actorUid, Option<DjfsUid> ownerUidO,
                                DjfsResource resource, String eventType) {
        MapF<String, String> data = Cf.hashMap();
        String ownerUid = actorUid.asString();
        if (ownerUidO.isPresent()) {
            ownerUid = ownerUidO.get().asString();
        }
        String albumItemType = "resource";
        if (!ownerUid.equals(actorUid.asString())) {
            albumItemType = "shared_resource";
        }
        ObjectId albumId = album.getId();
        String albumTitle = album.getTitle();
        MediaType mediaType = ((FileDjfsResource) resource).getMediaType().getOrElse(MediaType.UNKNOWN);
        data.put("event_type", eventType);
        data.put("uid", actorUid.asString());
        data.put("owner_uid", ownerUid);
        data.put("lenta_media_type", mediaType.getLentaMediaType());
        data.put("album_id", albumId.toString());
        data.put("album_title", albumTitle);
        data.put("album_item_type", albumItemType);
        data.put("resource_type", "file");
        data.put("resource_media_type", mediaType.getStringRepresentation());
        data.put("album_item_id", albumItem.getId().toHexString());
        data.put("resource_address", resource.getPath().getFullPath());
        data.put("resource_file_id", resource.getFileId().map(DjfsFileId::getValue).getOrElse(""));
        log(data);
    }

    public void log(AlbumAppendActivity activity) {
        Album album = activity.getAlbum();
        AlbumItem albumItem = activity.getAlbumItem();
        DjfsUid actorUid = activity.getPrincipal().getUidO().getOrElse(album.getUid());
        Option<ShareInfo> shareInfoO = activity.getShareInfo();
        Option<DjfsUid> ownerUid = Option.empty();
        if (shareInfoO.isPresent()) {
            ownerUid = Option.of(activity.getShareInfo().get().getOwnerUid());
        }
        DjfsResource resource = activity.getResource();
        // TODO: add support for shared files?
        formatAlbumLog(album, albumItem, actorUid, ownerUid, resource, "album-items-append");
    }

    public void log(AlbumItemRemoveActivity activity) {
        Album album = activity.getAlbum();
        AlbumItem albumItem = activity.getAlbumItem();
        DjfsUid actorUid = activity.getPrincipal().getUidO().getOrElse(album.getUid());
        Option<ShareInfo> shareInfoO = activity.getShareInfo();
        Option<DjfsUid> ownerUid = Option.empty();
        if (shareInfoO.isPresent()) {
            ownerUid = Option.of(activity.getShareInfo().get().getOwnerUid());
        }
        DjfsResource resource = activity.getResource();
        // TODO: add support for shared files?
        formatAlbumLog(album, albumItem, actorUid, ownerUid, resource, "album-items-remove");
    }

    public MapF<String, String> getBaseAlbumData(DjfsUid uid, Album album, String albumTitle, String eventType) {
        ObjectId albumId = album.getId();
        MapF<String, String> data = Cf.hashMap();
        data.put("uid", uid.asString());
        data.put("album_id", albumId.toString());
        data.put("album_title", albumTitle);
        data.put("event_type", eventType);
        return data;
    }

    public void log(AlbumSetAttrActivity activity) {
        Album album = activity.getAlbum();
        String albumTitle = activity.getAlbumTitle().getOrElse(album.getTitle());
        DjfsUid actorUid = activity.getPrincipal().getUidO().getOrElse(album.getUid());

        if (activity.getPrevAlbumTitle().isPresent()) {
            MapF<String, String> data = getBaseAlbumData(actorUid, album, albumTitle,"album-change-title");
            data.put("prev_album_title", activity.getPrevAlbumTitle().get());
            log(data);
        }
        if (activity.getCover().isPresent()) {
            log(getBaseAlbumData(actorUid, album, albumTitle, "album-change-cover"));
        }
        if (activity.getCoverOffsetY().isPresent()) {
            log(getBaseAlbumData(actorUid, album, albumTitle, "album-change-cover-offset"));
        }
        if (activity.getIsPublic().isPresent()) {
            MapF<String, String> data = getBaseAlbumData(actorUid, album, albumTitle, "album-change-publicity");
            String publicity = activity.getIsPublic().get().booleanValue() ? "1" : "0";
            data.put("album_is_public", publicity);
            log(data);
        }
    }

    public void log(AlbumRemoveActivity activity) {
        Album album = activity.getAlbum();
        DjfsUid actorUid = activity.getPrincipal().getUidO().getOrElse(album.getUid());
        MapF<String, String> data = Cf.hashMap();
        ObjectId albumId = album.getId();
        String albumTitle = album.getTitle();
        data.put("event_type", "album-remove");
        data.put("uid", actorUid.asString());
        data.put("album_id", albumId.toString());
        data.put("album_title", albumTitle);
        log(data);
    }

    public void log(CopyActivity activity) {
        MapF<String, String> data = Cf.hashMap();
        DjfsResource destination = activity.getDestination();
        DjfsUid destinationUid = destination.getUid();

        // todo: how to log system actions?
        DjfsUid actorUid = activity.getPrincipal().getUidO().getOrElse(destinationUid);

        SetF<DjfsUid> uids = Cf.hashSet(actorUid, destinationUid);
        if (activity.getDestinationShareInfo().isPresent()) {
            uids.addAll(activity.getDestinationShareInfo().get().allUids());
        }

        for (DjfsUid uid : uids) {
            data.put("event_type", "fs-copy");
            // todo: pass this value
            data.put("overwritten", "false");
            // todo: pass this value
            data.put("force", "false");

            data.put("resource_type", destination instanceof FolderDjfsResource ? "dir" : "file");
            data.put("resource_file_id", destination.getFileId().map(DjfsFileId::getValue).getOrElse(""));

            if (destination instanceof FileDjfsResource) {
                MediaType mediaType = ((FileDjfsResource) destination).getMediaType().getOrElse(MediaType.UNKNOWN);
                data.put("resource_media_type", mediaType.getStringRepresentation());
                data.put("lenta_media_type", mediaType.getLentaMediaType());
            }

            if (Objects.equals(activity.getSource().getUid(), uid)) {
                data.put("src_rawaddress", activity.getSource().getPath().getFullPath());
                data.put("src_folder_id", getFolderId(activity.getSource()));
            } else if (activity.getSourceShareInfo().isPresent()) {
                ShareInfo share = activity.getSourceShareInfo().get();
                if (share.getPermissions(uid).map(x -> x.hasPermission(FilesystemPermission.READ)).getOrElse(false)) {
                    Option<DjfsResourcePath> path = share.ownerPathToParticipantPath(activity.getSource().getPath(), uid);
                    if (path.isPresent()) {
                        data.put("src_rawaddress", path.get().getFullPath());
                    }
                    data.put("src_folder_id", getFolderId(activity.getSource()));
                }
            }

            if (Objects.equals(destinationUid, uid)) {
                data.put("tgt_rawaddress", destination.getPath().getFullPath());
                data.put("tgt_folder_id", getFolderId(destination));
            } else if (activity.getDestinationShareInfo().isPresent()) {
                ShareInfo share = activity.getDestinationShareInfo().get();
                if (share.getPermissions(uid).map(x -> x.hasPermission(FilesystemPermission.READ)).getOrElse(false)) {
                    Option<DjfsResourcePath> path = share.ownerPathToParticipantPath(destination.getPath(), uid);
                    if (path.isPresent()) {
                        data.put("tgt_rawaddress", path.get().getFullPath());
                    }
                    data.put("tgt_folder_id", getFolderId(destination));
                }
            }

            data.put("owner_uid", destinationUid.asString());
            data.put("uid", uid.asString());
            if (activity.getDestinationShareInfo().isPresent()) {
                data.put("user_uid", actorUid.asString());
            }

            log(data);
        }
    }

    public void log(MoveActivity activity) {
        MapF<String, String> data = Cf.hashMap();
        DjfsResource source = activity.getSource();
        DjfsResource destination = activity.getDestination();
        DjfsUid destinationUid = destination.getUid();
        DjfsUid actorUid = activity.getPrincipal().getUidO().getOrElse(destinationUid);

        // TODO: add support for shared folders

        if (!actorUid.equals(destinationUid)) {
            throw new NotImplementedException();
        }

        String eventType = "fs-move";
        if (activity.getDestination().getPath().getArea().equals(DjfsResourceArea.TRASH) &&
                !activity.getDestination().getPath().getArea().equals(activity.getSource().getPath().getArea())) {
            eventType = "fs-trash-append";
        }
        data.put("event_type", eventType);

        data.put("owner_uid", destinationUid.asString());
        data.put("uid", destinationUid.asString());

        data.put("force", Boolean.toString(activity.isForce()));
        data.put("overwritten", "false");  // todo: pass this value

        if (!(destination instanceof FolderDjfsResource)) {
            throw new NotImplementedException();
        }

        data.put("resource_type", "dir");
        data.put("resource_file_id", destination.getFileId().map(DjfsFileId::getValue).getOrElse(""));

        data.put("src_rawaddress", source.getPath().getFullPath());
        data.put("tgt_rawaddress", destination.getPath().getFullPath());

        data.put("src_folder_id", getFolderId(source));
        data.put("tgt_folder_id", getFolderId(destination));

        Option<Instant> creationTime = ((FolderDjfsResource) destination).getCreationTime();
        Option<Instant> uploadTime = destination.getUploadTime();

        data.put("tgt_ctime", creationTime.map(InstantUtils::toSeconds).map(x -> Integer.toString(x)).getOrElse(""));
        data.put("tgt_utime", uploadTime.map(InstantUtils::toSeconds).map(x -> Integer.toString(x)).getOrElse(""));

        log(data);
    }

    private String getFolderId(DjfsResource resource) {
        if (resource instanceof FolderDjfsResource) {
            return getResourceId(resource);
        }
        Option<FolderDjfsResource> parent = djfsResourceDao.find2Parent(resource);
        if (parent.isPresent()) {
            return getResourceId(parent.get());
        }
        return "";
    }

    private String getResourceId(DjfsResource resource) {
        return resource.getResourceId().map(DjfsResourceId::toString).getOrElse(resource.getPath().getFullPath());
    }

    protected void log(MapF<String, String> data) {
        logger.info(TskvUtils.formatTskvLine(data, false));
    }
}
