package ru.yandex.chemodan.app.djfs.core.index;

import java.util.UUID;

import com.mongodb.ReadPreference;
import lombok.RequiredArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.ActionContext;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsPrincipal;
import ru.yandex.chemodan.app.djfs.core.filesystem.Filesystem;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderDjfsResource;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfo;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfoManager;
import ru.yandex.chemodan.app.djfs.core.user.CheckBlocked;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserDao;
import ru.yandex.chemodan.app.djfs.core.user.UserData;
import ru.yandex.chemodan.app.djfs.core.user.UserNotInitializedException;
import ru.yandex.chemodan.util.retry.RetryManager;
import ru.yandex.chemodan.util.web.interceptors.WithThreadLocalCache;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestListParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.parse.BenderParser;
import ru.yandex.misc.bender.parse.BenderParserException;
import ru.yandex.misc.cache.tl.TlCache;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.web.servlet.HttpServletRequestX;


@ActionContainer
@RequiredArgsConstructor
public class IndexerActions {
    private static final int MAX_RESOURCES = 100;

    private static final ListF<DjfsResourceArea> areasToSearch =
            Cf.wrap(DjfsResourceArea.values()).filter(DjfsResourceArea::isSearchable);

    private final Filesystem filesystem;
    private final ShareInfoManager shareInfoManager;
    private final UserDao userDao;
    private final BazingaTaskManager bazingaTaskManager;
    private final BazingaManagerProperties bazingaTaskManagerProperties;
    private final IndexerManager indexerManager;
    private final BenderParser<IndexerBinaryData> binaryDataParser = new BenderMapper().createParser(IndexerBinaryData.class);
    private final BenderParser<IndexerComputerVisionData> cvDataParser = new BenderMapper().createParser(IndexerComputerVisionData.class);

    @CheckBlocked(httpStatusCode = HttpStatus.SC_451_UNAVAILABLE_FOR_LEGAL_REASONS)
    @WithThreadLocalCache
    @Path(value = "/v1/indexer/resources", methods = {HttpMethod.GET})
    public IndexerResourcesListPojo getResourcesByResourceIds(
            @RequestListParam("resourceId") ListF<String> rawResourceIds,
            @RequestParam("uid") String rawUid,
            @SpecialParam HttpServletRequestX req)
    {
        // just in case, check that it works and remove it
        TlCache.flush();

        if (rawResourceIds.size() > MAX_RESOURCES) {
            throw new IndexerGetResourcesTooManyResourcesException();
        }
        DjfsUid uid = DjfsUid.cons(rawUid, ActionContext.CLIENT_INPUT);
        DjfsPrincipal principal;
        ListF<DjfsResourceId> resourceIds = rawResourceIds.map(x -> DjfsResourceId.cons(x, ActionContext.CLIENT_INPUT));
        ListF<Tuple2<DjfsResource, ListF<UUID>>> resourcesWithParentIds;

        try {
            UserData user = UserData.getFromRequest(req);
            principal = DjfsPrincipal.cons(user);
            resourcesWithParentIds = filesystem.findResourcesWithParentIds(principal, resourceIds, areasToSearch,
                    Option.of(ReadPreference.primary()));
        } catch (UserNotInitializedException e) {
            throw new IndexerGetResourceNotFoundException();
        }

        ListF<IndexerResourcePojo> resultItems = Cf.linkedList();

        for (Tuple2<DjfsResource, ListF<UUID>> resourceWithParentIds : resourcesWithParentIds) {
            DjfsResource resource = resourceWithParentIds.get1();
            ListF<UUID> parentIds = resourceWithParentIds.get2();

            Option<ShareInfo> shareInfoO = shareInfoManager.get(resource, Option.of(ReadPreference.primary()));
            boolean isParticipantSharedResource = shareInfoO.isPresent() && !shareInfoO.get().getOwnerUid().equals(uid);

            if (resource instanceof FileDjfsResource) {
                UUID parentId  = parentIds.last();
                if (isParticipantSharedResource) {
                    ShareInfo shareInfo = shareInfoO.get();

                    DjfsResourcePath participantResourcePath =
                            shareInfo.ownerPathToParticipantPath(resource.getPath(), uid).get();
                    DjfsResourcePath participantSharedRootPath = shareInfo.getRootPath(uid).get();

                    if (participantResourcePath.getParent().equals(participantSharedRootPath)) {
                        parentId = filesystem.find(principal, participantSharedRootPath, Option.of(ReadPreference.primary())).map(DjfsResource::getId)
                                .getOrElse(parentId);
                    }
                }
                resultItems.add(new IndexerFilePojo(uid, (FileDjfsResource) resource, parentId, shareInfoO));
            } else if (resource instanceof FolderDjfsResource) {
                if (isParticipantSharedResource) {
                    ShareInfo shareInfo = shareInfoO.get();

                    DjfsResourcePath participantSharedRootPath = shareInfo.getRootPath(uid).get();
                    DjfsResourcePath ownerSharedRootPath = shareInfo.getGroupPath();

                    Option<Tuple2<DjfsResource, ListF<UUID>>> participantSharedRootWithParentIdsO =
                            filesystem.findResourceWithParentIds(principal, participantSharedRootPath);
                    if (!participantSharedRootWithParentIdsO.isPresent()) {
                        continue;
                    }
                    Tuple2<DjfsResource, ListF<UUID>> participantSharedRootWithParentIds =
                            participantSharedRootWithParentIdsO.get();

                    DjfsResource participantSharedRoot = participantSharedRootWithParentIds.get1();
                    ListF<UUID> participantSharedRootParentIds = participantSharedRootWithParentIds.get2();

                    // substitute the beginning of parentFids list with participant parent fids
                    parentIds = participantSharedRootParentIds
                            .plus(participantSharedRoot.getId())
                            .plus(parentIds.subList(ownerSharedRootPath.getAllParents().length() + 1, parentIds.length()));
                }

                resultItems.add(new IndexerFolderPojo(uid, (FolderDjfsResource) resource, parentIds, shareInfoO));
            } else {
                throw new NotImplementedException();
            }
        }

        return new IndexerResourcesListPojo(resultItems);
    }

    @Path(value = "/v1/indexer/set_aesthetics", methods = {HttpMethod.POST})
    public void saveAestheticsByResourceId(
            @RequestParam("uid") String rawUid,
            @RequestParam("resourceId") String rawResourceId,
            @SpecialParam HttpServletRequestX request)
    {
        IndexerComputerVisionData cvData;
        try {
            cvData = cvDataParser.parseJson(new String(request.getInputStreamX().readBytes()));
        } catch (BenderParserException e) {
            throw new IndexerInvalidBodyException("wrong format");
        } catch (IllegalArgumentException e) {
            throw new IndexerInvalidBodyException("cannot parse body");
        }

        if (!DjfsFileId.cons(cvData.getFileId()).equals(DjfsResourceId.cons(rawResourceId).getFileId())) {
            throw new IndexerInvalidFileIdException(cvData.getFileId());
        }
        new RetryManager().withRetryPolicy(
                bazingaTaskManagerProperties.getMaxRetries(),
                bazingaTaskManagerProperties.getRetryDelayMillis()
        ).run(() ->
            bazingaTaskManager.schedule(
                            new IndexerSaveAestheticsTask(rawUid, rawResourceId, cvData.getAesthetics())
                    )
        );
    }

    @Path(value = "/v1/indexer/set_dimensions", methods = {HttpMethod.POST})
    public void saveBinaryDataByResourceId(
            @RequestParam("uid") String rawUid,
            @RequestParam("resourceId") String rawResourceId,
            @SpecialParam HttpServletRequestX request
    )
    {
        ListF<IndexerBinaryData> binaryDataItems;
        try {
            binaryDataItems = binaryDataParser.parseListJson(new String(request.getInputStreamX().readBytes()));
        } catch (BenderParserException e) {
            throw new IndexerInvalidBodyException("wrong format");
        } catch (IllegalArgumentException e) {
            throw new IndexerInvalidBodyException("cannot parse body");
        }
        if (binaryDataItems.length() != 1) {
            throw new IndexerInvalidBodyException("length should be equal to 1, got = " + binaryDataItems.length());
        }

        DjfsUid uid = DjfsUid.cons(rawUid, ActionContext.CLIENT_INPUT);
        DjfsResourceId resourceId = DjfsResourceId.cons(rawResourceId, ActionContext.CLIENT_INPUT);
        IndexerBinaryData binaryData = binaryDataItems.first();

        if (!DjfsFileId.cons(binaryData.getFileId()).equals(resourceId.getFileId())) {
            throw new IndexerInvalidFileIdException(binaryData.getFileId());
        }

        if (!binaryData.getOrientation().isPresent()
                || !binaryData.getWidth().isPresent()
                || !binaryData.getHeight().isPresent()) {
            throw new IndexerInvalidDimensionsException();
        }

        try {
            indexerManager.setDimension(
                    uid, resourceId,
                    binaryData.getWidth().get(), binaryData.getHeight().get(), binaryData.getOrientation().get()
            );
        } catch (NotImplementedException e) {
            throw new IndexerUnsupportedUserException();
        }

        if (binaryData.getCoordinates().isPresent()) {
            new RetryManager().withRetryPolicy(
                    bazingaTaskManagerProperties.getMaxRetries(),
                    bazingaTaskManagerProperties.getRetryDelayMillis()
            ).run(() ->
                    bazingaTaskManager.schedule(
                            new IndexerSaveCoordinatesTask(rawUid, rawResourceId, binaryData.getCoordinates().get())
                    )
            );
        }
    }

    @Path(value = "/v1/indexer/fetch_extracted_data", methods = {HttpMethod.POST})
    public void fetchExtractedDataFromSearch(
            @RequestParam("uid") String rawUid,
            @RequestListParam("resourceId") ListF<String> rawResourceIds)
    {
        bazingaTaskManager.schedule(new IndexerFetchExtractedDataTask(rawUid, rawResourceIds));
    }
}
