package ru.yandex.chemodan.app.djfs.core.index;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.djfs.core.ActionContext;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.tasks.DjfsTaskQueueName;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserDao;
import ru.yandex.chemodan.app.djfs.core.user.UserIsBlockedException;
import ru.yandex.chemodan.bazinga.YcridOnetimeTaskSupport;
import ru.yandex.chemodan.bazinga.YcridTaskParameters;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleConstant;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;


public class IndexerFetchExtractedDataTask extends YcridOnetimeTaskSupport<IndexerFetchExtractedDataTask.Parameters> {
    private static final Logger logger = LoggerFactory.getLogger(IndexerFetchExtractedDataTask.class);

    private final UserDao userDao;
    private final IndexerManager indexerManager;

    public IndexerFetchExtractedDataTask(UserDao userDao, IndexerManager indexerManager) {
        super(Parameters.class);
        this.userDao = userDao;
        this.indexerManager = indexerManager;
    }

    public IndexerFetchExtractedDataTask(String uid, ListF<String> resourceIds) {
        super(new Parameters(uid, resourceIds));
        this.userDao = null;
        this.indexerManager = null;
    }

    @Override
    protected void doExecute(Parameters parameters, ExecutionContext context) {
        DjfsUid uid = DjfsUid.cons(parameters.uid, ActionContext.CLIENT_INPUT);

        ListF<DjfsResourceId> resourceIds = parameters.resourceIds
                .map(x -> DjfsResourceId.cons(x, ActionContext.CLIENT_INPUT));

        try {
            indexerManager.fetchExtractedDataFromDiskSearch(uid, resourceIds);
        } catch (UserIsBlockedException e) {
            // do nothing in case of blocked user
        }
    }

    @Override
    public TaskQueueName queueName() {
        return DjfsTaskQueueName.INDEXER_CALLBACK_TASKS;
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardSeconds(100);
    }

    @BenderBindAllFields
    static class Parameters extends YcridTaskParameters {
        private final String uid;
        private final ListF<String> resourceIds;

        Parameters(String uid, ListF<String> resourceIds) {
            this.uid = uid;
            this.resourceIds = resourceIds;
        }
    }

    @Override
    public ReschedulePolicy reschedulePolicy() {
        return new RescheduleConstant(Duration.standardMinutes(2), 10);
    }
}
