package ru.yandex.chemodan.app.djfs.core.index;

import com.mongodb.ReadPreference;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.djfs.core.client.DiskSearchHttpClient;
import ru.yandex.chemodan.app.djfs.core.client.DiskSearchResponse;
import ru.yandex.chemodan.app.djfs.core.client.DiskSearchResponseItem;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsPrincipal;
import ru.yandex.chemodan.app.djfs.core.filesystem.Filesystem;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserDao;
import ru.yandex.chemodan.app.djfs.core.user.UserData;
import ru.yandex.misc.geo.Coordinates;
import ru.yandex.misc.image.Dimension;

public class IndexerManager {
    private final Filesystem filesystem;
    private final DiskSearchHttpClient httpClient;
    private final UserDao userDao;

    public IndexerManager(Filesystem filesystem, DiskSearchHttpClient httpClient, UserDao userDao) {
        this.filesystem = filesystem;
        this.httpClient = httpClient;
        this.userDao = userDao;
    }

    public void fetchExtractedDataFromDiskSearch(DjfsUid uid, ListF<DjfsResourceId> resourceIds) {
        UserData user = userDao.findExistingAndNotBlocked(uid);
        DjfsPrincipal principal = DjfsPrincipal.cons(user);

        resourceIds = resourceIds.filter(x -> x.getUid().equals(uid));  // only for user resources

        ListF<DjfsResource> resources = filesystem.find(principal, resourceIds, DjfsResourceArea.ALL, Option.of(ReadPreference.primary()));
        ListF<FileDjfsResource> files =
                resources.filterByType(FileDjfsResource.class).filter(x -> x.getFileId().isPresent());

        DiskSearchResponse response = httpClient.getExtractedData(
                user.getUid(),
                files.map(DjfsResource::getFileId).filter(Option::isPresent).map(Option::get),
                user.isQuickMoveUser()
        );

        for (DiskSearchResponseItem item : response.getHitsArray()) {
            DjfsResourceId resourceId = DjfsResourceId.cons(user.getUid(), item.getFileId());

            Tuple2<Dimension, Integer> dimensionWithAngle =
                    convertIndexerDimension(item.getWidth(), item.getHeight(), item.getOrientation());

            if (item.getAesthetics().isPresent()) {
                filesystem.setAesthetics(principal, resourceId, item.getAesthetics().get(), true);
            }
            filesystem.setDimensionsWithAngle(principal, resourceId, dimensionWithAngle._1, dimensionWithAngle._2, true);
        }
    }

    public void setAesthetics(DjfsUid uid, DjfsResourceId resourceId, double aesthetics) {
        UserData user = userDao.findExistingAndNotBlocked(uid);
        DjfsPrincipal principal = DjfsPrincipal.cons(user);
        filesystem.setAesthetics(principal, resourceId, aesthetics, true);
    }

    public void setDimension(DjfsUid uid, DjfsResourceId resourceId, int width, int height,
            IndexerOrientation orientation)
    {
        UserData user = userDao.findExistingAndNotBlocked(uid);
        DjfsPrincipal principal = DjfsPrincipal.cons(user);
        Tuple2<Dimension, Integer> dimensionWithAngle = convertIndexerDimension(width, height, orientation);
        filesystem.setDimensionsWithAngle(principal, resourceId, dimensionWithAngle._1, dimensionWithAngle._2, true);
    }

    public void setCoordinates(DjfsUid uid, DjfsResourceId resourceId, Coordinates coordinates) {
        UserData user = userDao.findExistingAndNotBlocked(uid);
        DjfsPrincipal principal = DjfsPrincipal.cons(user);
        filesystem.setCoordinates(principal, resourceId, coordinates, true);
    }

    public static Tuple2<Dimension, Integer> convertIndexerDimension(int width, int height,
            IndexerOrientation orientation)
    {
        if (orientation == IndexerOrientation.LANDSCAPE) {
            return Tuple2.tuple(new Dimension(width, height), 0);
        }
        return Tuple2.tuple(new Dimension(height, width), 90);
    }
}
