package ru.yandex.chemodan.app.djfs.core.index;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderDjfsResource;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfo;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderParseSubclasses;
import ru.yandex.misc.bender.annotation.BenderPart;

@BenderBindAllFields
@BenderParseSubclasses({
        IndexerFilePojo.class,
        IndexerFolderPojo.class,
})
public abstract class IndexerResourcePojo {
    public final String id;
    @BenderPart(name = "resource_id", strictName = true)
    public final String resourceId;
    public final String name;
    public final String key;
    public final String uid;
    public final String type;
    public final String area;

    public final Boolean visible;
    public final Long version;

    @BenderPart(name = "shared_folder_version", strictName = true)
    public final Option<Long> sharedFolderVersion;

    public IndexerResourcePojo(DjfsUid requestUid, DjfsResource resource, Option<ShareInfo> shareInfo) {
        id = resource.getResourceId()
                .getOrThrow(() -> new UnableToBuildResponseFromResourceException(resource, "Missing fileId"))
                .getFileId().getValue();
        resourceId = resource.getResourceId()
                .getOrThrow(() -> new UnableToBuildResponseFromResourceException(resource, "Missing resourceId"))
                .toString();
        name = resource.getPath().getName();

        if (shareInfo.isPresent()) {
            key = shareInfo.get().ownerPathToParticipantPath(resource.getPath(), requestUid).get().getPath();
        } else {
            key = resource.getPath().getPath();
        }

        area = resource.getPath().getArea().rootFolderName;

        uid = resource.getUid().asString();

        if (resource instanceof FileDjfsResource) {
            type = "file";
        } else if (resource instanceof FolderDjfsResource) {
            type = "dir";
        } else {
            throw new NotImplementedException();
        }

        visible = resource.isVisible();

        Long version = resource.getVersion()
                .getOrThrow(() -> new UnableToBuildResponseFromResourceException(resource, "Missing version"));
        this.version = version;

        if (shareInfo.isPresent()) {
            Option<Long> groupLinkBaseVersionO = shareInfo.get().getGroupLinkBaseVersion(requestUid);
            if (groupLinkBaseVersionO.isPresent()) {
                sharedFolderVersion = Option.of(groupLinkBaseVersionO.get() + version);
            } else {
                sharedFolderVersion = Option.empty();
            }
        } else {
            sharedFolderVersion = Option.empty();
        }
    }
}
