package ru.yandex.chemodan.app.djfs.core.index;

import org.joda.time.Duration;

import ru.yandex.chemodan.app.djfs.core.ActionContext;
import ru.yandex.chemodan.app.djfs.core.album.GeoAlbumCoverUpdateTask;
import ru.yandex.chemodan.app.djfs.core.album.GeoAlbumGenerationProperties;
import ru.yandex.chemodan.app.djfs.core.album.GeoAlbumManager;
import ru.yandex.chemodan.app.djfs.core.album.worker.DjfsAlbumsTaskManager;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.tasks.DjfsTaskQueueName;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserIsBlockedException;
import ru.yandex.chemodan.bazinga.YcridOnetimeTaskSupport;
import ru.yandex.chemodan.bazinga.YcridTaskParameters;
import ru.yandex.chemodan.util.retry.RetryManager;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.CompoundReschedulePolicy;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleConstant;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleExponential;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;


public class IndexerSaveAestheticsTask extends YcridOnetimeTaskSupport<IndexerSaveAestheticsTask.Parameters> {
    private static final Logger logger = LoggerFactory.getLogger(IndexerSaveAestheticsTask.class);

    private final IndexerManager indexerManager;
    private final DjfsAlbumsTaskManager djfsAlbumsTaskManager;
    private final BazingaTaskManager bazingaTaskManager;
    private final BazingaManagerProperties bazingaTaskManagerProperties;
    private final GeoAlbumGenerationProperties geoAlbumGenerationProperties;
    private final GeoAlbumManager geoAlbumManager;

    private final DynamicProperty<Boolean> disableLinearization = new DynamicProperty<>(
            "disk-djfs-albums-disable-linearization", true);

    public IndexerSaveAestheticsTask(IndexerManager indexerManager, DjfsAlbumsTaskManager djfsAlbumsTaskManager,
            BazingaTaskManager bazingaTaskManager, BazingaManagerProperties bazingaTaskManagerProperties,
            GeoAlbumGenerationProperties geoAlbumGenerationProperties, GeoAlbumManager geoAlbumManager)
    {
        super(Parameters.class);
        this.indexerManager = indexerManager;
        this.djfsAlbumsTaskManager = djfsAlbumsTaskManager;
        this.bazingaTaskManager = bazingaTaskManager;
        this.bazingaTaskManagerProperties = bazingaTaskManagerProperties;
        this.geoAlbumGenerationProperties = geoAlbumGenerationProperties;
        this.geoAlbumManager = geoAlbumManager;
    }

    public IndexerSaveAestheticsTask(String uid, String resourceId, double aesthetics) {
        super(new Parameters(uid, resourceId, aesthetics));
        this.indexerManager = null;
        this.bazingaTaskManager = null;
        this.djfsAlbumsTaskManager = null;
        this.bazingaTaskManagerProperties = null;
        this.geoAlbumGenerationProperties = null;
        this.geoAlbumManager = null;
    }

    @Override
    protected void doExecute(Parameters parameters, ExecutionContext context) {
        DjfsUid uid = DjfsUid.cons(parameters.uid, ActionContext.CLIENT_INPUT);
        DjfsResourceId resourceId = DjfsResourceId.cons(parameters.resourceId, ActionContext.CLIENT_INPUT);
        try {
            indexerManager.setAesthetics(uid, resourceId, parameters.aesthetics);
            if (geoAlbumGenerationProperties.getGeoAlbumGenerationEnabled().get()) {
                appendGeoAlbumCoverUpdateTask(uid, resourceId);
            }
        } catch (UserIsBlockedException e) {
            // do nothing in case of blocked user
        }
    }

    private void appendGeoAlbumCoverUpdateTask(DjfsUid uid, DjfsResourceId resourceId) {
        if (geoAlbumManager.hasGeoAlbums(uid)) {
            if (disableLinearization.get()) {
                new RetryManager().withRetryPolicy(
                        bazingaTaskManagerProperties.getMaxRetries(),
                        bazingaTaskManagerProperties.getRetryDelayMillis()
                ).run(() ->
                        bazingaTaskManager.schedule(
                                new GeoAlbumCoverUpdateTask(uid.asString(), resourceId.getValue())
                        )
                );
            } else {
                djfsAlbumsTaskManager.schedule(
                        new GeoAlbumCoverUpdateTask(uid.asString(), resourceId.getValue()),
                        bazingaTaskManagerProperties.getMaxRetries(),
                        bazingaTaskManagerProperties.getRetryDelayMillis()
                );
            }
        }
    }

    @Override
    public TaskQueueName queueName() {
        return DjfsTaskQueueName.INDEXER_CALLBACK_TASKS;
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardSeconds(100);
    }

    @BenderBindAllFields
    static class Parameters extends YcridTaskParameters {
        private final String uid;
        private final String resourceId;
        private final double aesthetics;

        Parameters(String uid, String resourceId, double aesthetics) {
            this.uid = uid;
            this.resourceId = resourceId;
            this.aesthetics = aesthetics;
        }
    }

    @Override
    public ReschedulePolicy reschedulePolicy() {
        return new CompoundReschedulePolicy(
            new RescheduleConstant(Duration.standardMinutes(1), 10),
            new RescheduleExponential(Duration.standardMinutes(1), 10)
        );
    }
}
