package ru.yandex.chemodan.app.djfs.core.index;

import org.joda.time.Duration;

import ru.yandex.chemodan.app.djfs.core.ActionContext;
import ru.yandex.chemodan.app.djfs.core.album.GeoAlbumGenerationProperties;
import ru.yandex.chemodan.app.djfs.core.album.GeoAlbumManager;
import ru.yandex.chemodan.app.djfs.core.album.GeoAlbumResourceProcessingTask;
import ru.yandex.chemodan.app.djfs.core.album.worker.DjfsAlbumsTaskManager;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.tasks.DjfsTaskQueueName;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserIsBlockedException;
import ru.yandex.chemodan.bazinga.YcridOnetimeTaskSupport;
import ru.yandex.chemodan.bazinga.YcridTaskParameters;
import ru.yandex.chemodan.util.retry.RetryManager;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.CompoundReschedulePolicy;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleConstant;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleExponential;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.geo.Coordinates;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;


public class IndexerSaveCoordinatesTask extends YcridOnetimeTaskSupport<IndexerSaveCoordinatesTask.Parameters> {
    private static final Logger logger = LoggerFactory.getLogger(IndexerSaveCoordinatesTask.class);

    private final IndexerManager indexerManager;
    private final DjfsAlbumsTaskManager djfsAlbumsTaskManager;
    private final BazingaTaskManager bazingaTaskManager;
    private final BazingaManagerProperties bazingaTaskManagerProperties;
    private final GeoAlbumGenerationProperties geoAlbumGenerationProperties;
    private final GeoAlbumManager geoAlbumManager;

    private final DynamicProperty<Boolean> disableLinearization = new DynamicProperty<>(
            "disk-djfs-albums-disable-linearization", true);

    public IndexerSaveCoordinatesTask(IndexerManager indexerManager, DjfsAlbumsTaskManager djfsAlbumsTaskManager,
            BazingaTaskManager bazingaTaskManager, BazingaManagerProperties bazingaManagerProperties,
            GeoAlbumGenerationProperties geoAlbumGenerationProperties, GeoAlbumManager geoAlbumManager)
    {
        super(Parameters.class);
        this.indexerManager = indexerManager;
        this.djfsAlbumsTaskManager = djfsAlbumsTaskManager;
        this.bazingaTaskManager = bazingaTaskManager;
        this.bazingaTaskManagerProperties = bazingaManagerProperties;
        this.geoAlbumGenerationProperties = geoAlbumGenerationProperties;
        this.geoAlbumManager = geoAlbumManager;
    }

    public IndexerSaveCoordinatesTask(String uid, String resourceId, Coordinates coordinates) {
        super(new Parameters(uid, resourceId, coordinates.getLatitude(), coordinates.getLongitude()));
        this.indexerManager = null;
        this.djfsAlbumsTaskManager = null;
        this.bazingaTaskManager = null;
        this.bazingaTaskManagerProperties = null;
        this.geoAlbumGenerationProperties = null;
        this.geoAlbumManager = null;
    }

    @Override
    protected void doExecute(Parameters parameters, ExecutionContext context) {
        DjfsUid uid = DjfsUid.cons(parameters.uid, ActionContext.CLIENT_INPUT);
        DjfsResourceId resourceId = DjfsResourceId.cons(parameters.resourceId, ActionContext.CLIENT_INPUT);
        try {
            Coordinates coordinates = new Coordinates(parameters.latitude, parameters.longitude);
            indexerManager.setCoordinates(uid, resourceId, coordinates);
            if (geoAlbumGenerationProperties.getGeoAlbumGenerationEnabled().get()) {
                appendGeoAlbumGenerationTask(uid, resourceId);
            }
        } catch (UserIsBlockedException e) {
            // do nothing in case of blocked user
        }
    }

    private void appendGeoAlbumGenerationTask(DjfsUid uid, DjfsResourceId resourceId) {
        if (geoAlbumManager.hasGeoAlbums(uid)) {
            if (disableLinearization.get()) {
                new RetryManager().withRetryPolicy(
                        bazingaTaskManagerProperties.getMaxRetries(),
                        bazingaTaskManagerProperties.getRetryDelayMillis()
                ).run(() ->
                        bazingaTaskManager.schedule(
                                new GeoAlbumResourceProcessingTask(uid.asString(), resourceId.getValue())
                        )
                );
            } else {
                djfsAlbumsTaskManager.schedule(
                        new GeoAlbumResourceProcessingTask(uid.asString(), resourceId.getValue()),
                        bazingaTaskManagerProperties.getMaxRetries(),
                        bazingaTaskManagerProperties.getRetryDelayMillis()
                );
            }
        }
    }

    @Override
    public TaskQueueName queueName() {
        return DjfsTaskQueueName.INDEXER_CALLBACK_TASKS;
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardSeconds(100);
    }

    @BenderBindAllFields
    static class Parameters extends YcridTaskParameters {
        private final String uid;
        private final String resourceId;
        private final double latitude;
        private final double longitude;

        Parameters(String uid, String resourceId, double latitude, double longitude) {
            this.uid = uid;
            this.resourceId = resourceId;
            this.latitude = latitude;
            this.longitude = longitude;
        }
    }

    @Override
    public ReschedulePolicy reschedulePolicy() {
        return new CompoundReschedulePolicy(
            new RescheduleConstant(Duration.standardMinutes(1), 10),
            new RescheduleExponential(Duration.standardMinutes(1), 10)
        );
    }
}
