package ru.yandex.chemodan.app.djfs.core.index;

import java.util.concurrent.ExecutorService;

import lombok.RequiredArgsConstructor;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.EventManager;
import ru.yandex.chemodan.app.djfs.core.filesystem.CopyActivity;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsPrincipal;
import ru.yandex.chemodan.app.djfs.core.filesystem.MoveActivity;
import ru.yandex.chemodan.app.djfs.core.filesystem.event.FolderCreatedEvent;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.MediaType;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfo;
import ru.yandex.chemodan.app.djfs.core.share.event.UserLeftGroupEvent;
import ru.yandex.chemodan.app.djfs.core.user.UserDao;
import ru.yandex.chemodan.app.djfs.core.user.UserData;
import ru.yandex.chemodan.app.djfs.core.user.UserType;
import ru.yandex.chemodan.app.djfs.core.util.CeleryJobUtils;
import ru.yandex.chemodan.app.djfs.core.util.ExecutorServiceUtils;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.chemodan.app.djfs.core.util.JsonUtils;
import ru.yandex.chemodan.queller.celery.job.CeleryJob;
import ru.yandex.chemodan.queller.worker.CeleryTaskManager;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author eoshch
 */
@RequiredArgsConstructor
public class SearchPushGenerator {
    public static TaskId TASK_ID =
            new TaskId("mpfs.core.job_handlers.indexer.handle_notify_search_indexer");
    public static TaskId PHOTOSLICE_TASK_ID =
            new TaskId("mpfs.core.job_handlers.indexer.handle_notify_search_photoslice_indexer");
    public static TaskId SHARED_TASK_ID =
            new TaskId("mpfs.core.job_handlers.indexer.handle_notify_search_indexer_group");
    public static TaskId SHARED_PHOTOSLICE_TASK_ID =
            new TaskId("mpfs.core.job_handlers.indexer.handle_search_photoslice_indexer_group");

    private final ExecutorService executorService;
    private final CeleryTaskManager celeryTaskManager;
    private final UserDao userDao;
    private final SearchPushGeneratorProperties properties;

    // todo: don't forget to support search reindex (action: update -> modify)

    // todo: use sendPush
    @Deprecated
    @EventManager.EventHandler
    public void onFolderCreated(FolderCreatedEvent event) {
        Option<UserData> user = userDao.find(event.folder.getUid());
        if (!user.filter(userData -> userData.getType().map(type -> type == UserType.STANDARD).getOrElse(Boolean.FALSE)).isPresent()) {
            return;
        }
        if (event.folder.getPath().getDepth() >= properties.getMaxResourceDepth()) {
            return;
        }

        Option<ShareInfo> shareInfoO = event.getShareInfoO();

        JsonUtils.JsonObjectBuilder kwargsBuilder = JsonUtils.objectBuilder();
        if (shareInfoO.isPresent()) {
            kwargsBuilder.add("gid", shareInfoO.get().getGroupId());
        }

        // todo: use bender?
        kwargsBuilder.addArray("data", JsonUtils.objectBuilder()
                .add("uid", event.actor.asLong())
                .add("shared_folder_owner", shareInfoO.map(x -> x.getGroupPath().getUid().asString()).getOrNull())
                .add("visible", event.folder.isVisible() ? 1 : 0)
                .add("file_id", event.folder.getResourceId().map(DjfsResourceId::getFileId).map(
                        DjfsFileId::getValue).getOrNull())
                .add("resource_id", event.folder.getResourceId().map(DjfsResourceId::toString).getOrNull())
                .add("mtime", event.folder.getModificationTime().map(InstantUtils::toSeconds).getOrNull())
                .add("operation", "mkdir")
                .add("id", event.folder.getPath().getPath())
                .add("name", event.folder.getPath().getName())
                .add("ctime", event.folder.getCreationTime().map(InstantUtils::toSeconds).getOrNull())
                .add("folder_type", event.folder.getFolderType().map(x -> x.name().toLowerCase()).getOrNull())
                .add("version", event.folder.getVersion().getOrNull())
                .add("real_resource_version", event.folder.getVersion().getOrElse(0L))
                .add("action", "modify")
                .add("type", "dir")
                .toJsonObject());

        TaskId taskId = shareInfoO.map(x -> SHARED_TASK_ID).getOrElse(TASK_ID);
        CeleryJob celeryJob = CeleryJobUtils.create(taskId, kwargsBuilder.toMap());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));
    }

    @EventManager.EventHandler
    public void onUserLeftGroup(UserLeftGroupEvent event) {
        // todo: call MPFS to reindex user
    }

    public void sendPush(CopyActivity activity) {
        sendPush(activity.getPrincipal(), OperationType.COPY, activity.getDestination(),
                activity.getDestinationShareInfo(), Option.of(activity.getSource()), false);
    }

    public void sendPush(MoveActivity activity) {
        sendPush(activity.getPrincipal(), OperationType.MOVE, activity.getDestination(),
                Option.empty(), Option.empty(), true);
    }

    private void sendPush(DjfsPrincipal principal, OperationType type, DjfsResource resource,
            Option<ShareInfo> shareInfo, Option<DjfsResource> source, boolean forcePhotosliceUpdate)
    {
        Option<UserData> user = userDao.find(resource.getUid());
        if (!user.filter(userData -> userData.getType().map(userType -> userType == UserType.STANDARD).getOrElse(Boolean.FALSE)).isPresent()) {
            return;
        }
        if (resource.getPath().getDepth() >= properties.getMaxResourceDepth()) {
            return;
        }

        JsonUtils.JsonObjectBuilder kwargsBuilder = JsonUtils.objectBuilder();
        if (shareInfo.isPresent()) {
            kwargsBuilder.add("gid", shareInfo.get().getGroupId());
        }

        JsonUtils.JsonObjectBuilder dataBuilder = JsonUtils.objectBuilder()
                .add("action", type.action())
                .add("uid", principal.getUidO().getOrElse(resource.getUid()).asLong())
                .add("id", resource.getPath().getPath())
                .add("name", resource.getPath().getName())
                .add("file_id", resource.getResourceId().map(x -> x.getFileId().getValue()).getOrNull())
                .add("resource_id", resource.getResourceId().map(DjfsResourceId::toString).getOrNull())
                .add("visible", resource.isVisible() ? 1 : 0)
                .add("ctime", resource.getCreationTimeO().map(InstantUtils::toSeconds).getOrNull())
                .add("mtime", resource.getModificationTimeO().map(InstantUtils::toSeconds).getOrNull())
                .add("version", resource.getVersion().getOrElse(0L))
                .add("real_resource_version", resource.getVersion().getOrElse(0L));

        if (resource instanceof FileDjfsResource) {
            FileDjfsResource file = (FileDjfsResource) resource;
            dataBuilder.add("type", "file")
                    .add("operation", type.forFile())
                    .add("stid", file.getFileStid())
                    .add("size", file.getSize())
                    .add("md5", file.getMd5())
                    .add("mediatype", file.getMediaType().map(MediaType::getStringRepresentation).getOrNull())
                    .add("mimetype", file.getMimetype().getOrNull())
                    .add("external_url", file.getExternalUrl().getOrNull());

            // https://st.yandex-team.ru/CHEMODAN-38939
            if (file.getFotkiTags().isPresent()) {
                dataBuilder.add("fotki_tags", String.join("\n", StringUtils.split(file.getFotkiTags().get(), ",")));
            } else {
                dataBuilder.add("etime", file.getExifTime().map(InstantUtils::toSeconds).getOrNull());
            }

        } else {
            FolderDjfsResource folder = (FolderDjfsResource) resource;
            dataBuilder.add("type", "dir")
                    .add("operation", type.forFolder())
                    .add("folder_type", folder.getFolderType().map(x -> x.name().toLowerCase()).getOrNull())
                    .add("shared_folder_owner", shareInfo.map(x -> x.getGroupPath().getUid().asString()).getOrNull());
        }

        kwargsBuilder.addArray("data", dataBuilder
                .toJsonObject());

        TaskId taskId;
        if (resource.hasPhotosliceTime() || (source.isPresent() && source.get().hasPhotosliceTime()) ||
                forcePhotosliceUpdate)
        {
            if (shareInfo.isPresent()) {
                taskId = SHARED_PHOTOSLICE_TASK_ID;
            } else {
                taskId = PHOTOSLICE_TASK_ID;
            }
        } else {
            if (shareInfo.isPresent()) {
                taskId = SHARED_TASK_ID;
            } else {
                taskId = TASK_ID;
            }
        }

        CeleryJob celeryJob = CeleryJobUtils.create(taskId, kwargsBuilder.toMap());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));
    }

    private enum OperationType {
        CREATE {
            @Override
            protected String action() {
                return "modify";
            }

            @Override
            protected String forFile() {
                return "checkme_mkfile";
            }

            @Override
            protected String forFolder() {
                return "mkdir";
            }
        },
        COPY {
            @Override
            protected String action() {
                return "modify";
            }

            @Override
            protected String forFile() {
                return "copy_resource";
            }

            @Override
            protected String forFolder() {
                return "copy_resource";
            }
        },
        MOVE {
            @Override
            protected String action() {
                return "modify";
            }

            @Override
            protected String forFile() {
                return "move_resource";
            }

            @Override
            protected String forFolder() {
                return "move_resource";
            }
        },
        ;

        protected abstract String action();

        protected abstract String forFile();

        protected abstract String forFolder();
    }
}
