package ru.yandex.chemodan.app.djfs.core.legacy;

import java.util.List;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.mongodb.ReadPreference;
import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.djfs.core.ActionContext;
import ru.yandex.chemodan.app.djfs.core.album.AlbumDao;
import ru.yandex.chemodan.app.djfs.core.album.PersonalAlbumManager;
import ru.yandex.chemodan.app.djfs.core.db.mongo.DjfsBenderFactory;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsPrincipal;
import ru.yandex.chemodan.app.djfs.core.filesystem.Filesystem;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.DjfsNotImplementedException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.ResourceExistsException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.ResourceNotFoundException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.UnsupportedResourceTypeException;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DefaultFolder;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.exception.InvalidDjfsResourceAreaException;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.exception.InvalidDjfsResourcePathException;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.move.MoveCallbacks;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.move.MoveOperation;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.trashappend.TrashAppendOperation;
import ru.yandex.chemodan.app.djfs.core.history.EventHistoryLogger;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyAreaNotFoundException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyMd5UnsupportedException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyMpfsExceptionOrigin;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyPathErrorException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyPreconditionFailed;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyServiceNotFoundException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.mkdir.LegacyMkdirResourceExistsException;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.DefaultFolderPojo;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.DefaultFoldersFullPojo;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.DefaultFoldersOnlyPathPojo;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.FormattingContext;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.PublicInfoPojo;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.ReadResourceEndpoint;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.ResourcePojo;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.ResourcePojoBuilder;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.UserPojo;
import ru.yandex.chemodan.app.djfs.core.legacy.web.TranslateExceptionToLegacy;
import ru.yandex.chemodan.app.djfs.core.notification.XivaPushGenerator;
import ru.yandex.chemodan.app.djfs.core.operations.Operation;
import ru.yandex.chemodan.app.djfs.core.operations.OperationDao;
import ru.yandex.chemodan.app.djfs.core.publication.PublicAddress;
import ru.yandex.chemodan.app.djfs.core.publication.PublicationManager;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfoManager;
import ru.yandex.chemodan.app.djfs.core.user.CheckBlocked;
import ru.yandex.chemodan.app.djfs.core.user.ClientInputDataProcessor;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserDao;
import ru.yandex.chemodan.app.djfs.core.user.UserData;
import ru.yandex.chemodan.app.djfs.core.user.UserLocale;
import ru.yandex.chemodan.app.djfs.core.user.UserNotInitializedException;
import ru.yandex.chemodan.app.djfs.core.util.CeleryJobUtils;
import ru.yandex.chemodan.app.djfs.core.web.JsonStringResult;
import ru.yandex.chemodan.queller.celery.job.CeleryJob;
import ru.yandex.chemodan.queller.worker.CeleryTaskManager;
import ru.yandex.chemodan.util.web.OkPojo;
import ru.yandex.chemodan.util.web.interceptors.WithThreadLocalCache;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.IllegalParameterException;
import ru.yandex.commune.a3.action.parameter.bind.BoundJsonListByBender;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestListParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.bender.serialize.BenderJsonSerializer;
import ru.yandex.misc.cache.tl.TlCache;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

import static ru.yandex.chemodan.app.djfs.core.legacy.formatting.ReadResourceEndpoint.BULK_INFO_BY_RESOURCE_IDS;


/**
 * @author eoshch
 */
@ActionContainer
@RequiredArgsConstructor
public class LegacyFilesystemActions {

    public static final ObjectMapper mapper = new ObjectMapper();

    private static final SetF<String> VALID_TLDS = Cf.set("ru", "tr", "uk", "en");

    private final Filesystem filesystem;
    private final OperationDao operationDao;
    private final CeleryTaskManager celeryTaskManager;
    private final EventHistoryLogger eventHistoryLogger;
    private final ShareInfoManager shareInfoManager;
    private final UserDao userDao;
    private final AlbumDao albumDao;
    private final PersonalAlbumManager personalAlbumManager;
    private final ResourcePojoBuilder resourcePojoBuilder;
    private final XivaPushGenerator xivaPushGenerator;
    private final PublicationManager publicationManager;
    private final ClientInputDataProcessor clientInputDataProcessor;
    private final boolean isBlockingsCheckByHashEnabled;

    private final BenderJsonSerializer<ResourcePojo> resourcePojoSerializer =
            DjfsBenderFactory.createForJson(ResourcePojo.class).getSerializer();
    private final BenderJsonSerializer<PublicInfoPojo> publicPojoSerializer =
            DjfsBenderFactory.createForJson(PublicInfoPojo.class).getSerializer();

    private final SetF<DjfsResourceArea> bulkInfoAllowedAreas = Cf.set(
            DjfsResourceArea.DISK,
            DjfsResourceArea.ATTACH,
            DjfsResourceArea.TRASH,
            DjfsResourceArea.PHOTOUNLIM,
            DjfsResourceArea.NOTES
    );
    private final Language defaultLocale = Language.RUSSIAN;

    // todo: implement converter for DjfsUid ru.yandex.chemodan.app.commentaries.web.bind.ConverterToCommentId
    @Path(value = "/legacy/json/mkdir", methods = {HttpMethod.GET, HttpMethod.POST})
    @Path(value = "/legacy/desktop/mkdir", methods = {HttpMethod.GET, HttpMethod.POST})
    @TranslateExceptionToLegacy(origin = LegacyMpfsExceptionOrigin.MKDIR)
    public OkPojo mkdir(@RequestParam("uid") String uid, @RequestParam("path") String path,
            @RequestParam("force") Option<Integer> force)
    {
        if (force.isPresent() && force.get() != 0) {
            throw new IllegalParameterException("force");
        }

        DjfsResourcePath djfsResourcePath;
        try {
            djfsResourcePath = DjfsResourcePath.cons(uid, path);
        } catch (InvalidDjfsResourceAreaException e) {
            throw new LegacyAreaNotFoundException(e);
        } catch (InvalidDjfsResourcePathException e) {
            throw new LegacyPathErrorException(e);
        }

        if (djfsResourcePath.isAreaRoot()) {
            throw new LegacyPathErrorException(new InvalidDjfsResourcePathException("Area root not allowed",
                    djfsResourcePath));
        }

        try {
            filesystem.createFolder(DjfsPrincipal.cons(DjfsUid.cons(uid)), djfsResourcePath);
            return new OkPojo();
        } catch (ResourceExistsException e) {
            Option<DjfsResourcePath> availablePath = filesystem.getAvailableAutosuffixedPath(djfsResourcePath);
            throw new LegacyMkdirResourceExistsException(availablePath, e);
        }
    }

    @Path(value = "/legacy/json/move", methods = {HttpMethod.GET, HttpMethod.POST})
    @CheckBlocked
    @TranslateExceptionToLegacy(origin = LegacyMpfsExceptionOrigin.MOVE)
    public Object move(
            @RequestParam("uid") String rawUid,
            @RequestParam("src") String srcPath,
            @RequestParam("dst") String dstPath,
            @RequestParam("force") Option<Integer> force,
            @RequestParam("callback") Option<String> callback,  // nobody calls sync move with callcack
            @RequestParam("getLentaBlockId") Option<Integer> getLentaBlockId,  // no longer needed
            @RequestParam("returnStatus") Option<Integer> returnStatus,
            @RequestParam("checkHidsBlockings") Option<Boolean> checkHidsBlockings)
    {
        // TODO: add process of checkHidsBlockings flag when you start work on move for shared folders

        DjfsUid uid = DjfsUid.cons(rawUid, ActionContext.CLIENT_INPUT);
        DjfsResourcePath sourcePath = DjfsResourcePath.cons(srcPath);
        DjfsResourcePath destinationPath = DjfsResourcePath.cons(dstPath);
        boolean isForceMove = force.getOrElse(0) == 1;

        filesystem.moveSingleResource(DjfsPrincipal.cons(uid), sourcePath, destinationPath, isForceMove,
                MoveCallbacks.defaultWithLogging(eventHistoryLogger));

        if (returnStatus.getOrElse(0) == 1) {
            return new OkPojo();
        }
        return new JsonStringResult("\"\"");  // yes, that's what mpfs send to the client in that case
    }

    @Path(value = "/legacy/json/async_move", methods = {HttpMethod.GET, HttpMethod.POST})
    @CheckBlocked
    @TranslateExceptionToLegacy(origin = LegacyMpfsExceptionOrigin.MOVE)
    public AsyncOperationResultPojo asyncMove(
            @RequestParam("uid") String rawUid,
            @RequestParam("src") String srcPath,
            @RequestParam("dst") String dstPath,
            @RequestParam("force") Option<Integer> force,
            @RequestParam("callback") Option<String> callback,
            @SpecialParam HttpServletRequestX req)
    {
        DjfsUid uid = DjfsUid.cons(rawUid, ActionContext.CLIENT_INPUT);
        DjfsResourcePath sourcePath = DjfsResourcePath.cons(srcPath);
        DjfsResourcePath destinationPath = DjfsResourcePath.cons(dstPath);
        boolean isForceMove = force.getOrElse(0) == 1;

        if (!filesystem.isMoveImplemented(sourcePath, destinationPath, isForceMove)) {
            throw new DjfsNotImplementedException();
        }

        long userDiskVersion = UserData.getFromRequest(req).getVersion().getOrElse(0L);
        Option<DjfsResource> sourceO = filesystem.find(DjfsPrincipal.cons(uid), sourcePath, Option.of(ReadPreference.primary()));
        if (!sourceO.isPresent()) {
            throw new ResourceNotFoundException(sourcePath);
        }

        Operation operation = MoveOperation.create(uid, sourcePath, sourceO.get(), destinationPath, isForceMove,
                callback.getOrElse(""), userDiskVersion);
        operationDao.insert(operation);

        CeleryJob celeryJob = CeleryJobUtils.create(MoveOperation.TASK_ID, operation.getUid(), operation.getId());
        celeryTaskManager.submit(celeryJob);
        xivaPushGenerator.sendOperationCreatedPush(uid, operation);

        // TODO: fail operation simultaneously in case of submitting failes
        // TODO: if simultaneous fail failed as well -> submit async task to fail operation

        return new AsyncOperationResultPojo(operation, userDiskVersion);
    }

    @Path(value = "/legacy/json/trash_append", methods = {HttpMethod.GET, HttpMethod.POST})
    @CheckBlocked
    @TranslateExceptionToLegacy(origin = LegacyMpfsExceptionOrigin.MOVE)
    public TrashAppendResultPojo trashAppend(
            @RequestParam("uid") String rawUid,
            @RequestParam("path") String rawFullPath,
            @RequestParam("md5") String md5)  // nobody calls sync trashAppend with md5
    {
        DjfsUid uid = DjfsUid.cons(rawUid, ActionContext.CLIENT_INPUT);
        DjfsResourcePath path = DjfsResourcePath.cons(rawFullPath);

        DjfsResource removedResource = filesystem.trashAppendResource(DjfsPrincipal.cons(uid), path,
                MoveCallbacks.defaultWithLogging(eventHistoryLogger));

        // todo: recreate yateam folder if it was removed

        return new TrashAppendResultPojo(path, removedResource.getPath());
    }

    @Path(value = "/legacy/json/async_trash_append", methods = {HttpMethod.GET, HttpMethod.POST})
    @CheckBlocked
    @TranslateExceptionToLegacy(origin = LegacyMpfsExceptionOrigin.MOVE)
    public AsyncOperationResultPojo asyncTrashAppend(
            @RequestParam("uid") String rawUid,
            @RequestParam("path") String rawFullPath,
            @RequestParam("md5") Option<String> md5,
            @RequestParam("callback") Option<String> callback,
            @SpecialParam HttpServletRequestX req)
    {
        DjfsUid uid = DjfsUid.cons(rawUid, ActionContext.CLIENT_INPUT);
        DjfsResourcePath path = DjfsResourcePath.cons(rawFullPath);
        DjfsPrincipal principal = DjfsPrincipal.cons(uid);

        long userDiskVersion = UserData.getFromRequest(req).getVersion().getOrElse(0L);

        if (md5.isPresent() && !md5.get().isEmpty()) {
            Option<DjfsResource> resource = filesystem.find(principal, path, Option.of(ReadPreference.primary()));
            if (!resource.isPresent()) {
                throw new ResourceNotFoundException(path);
            }

            if (!(resource.get() instanceof FileDjfsResource)) {
                throw new LegacyMd5UnsupportedException(
                        new UnsupportedResourceTypeException("FileDjfsResource expected"));
            }

            FileDjfsResource file = (FileDjfsResource) resource.get();
            if (!file.getMd5().equals(md5.get())) {
                throw new LegacyPreconditionFailed("Resource md5 is different: " + file.getMd5());
            }
        }

        if (!filesystem.isTrashAppendImplemented(path)) {
            throw new DjfsNotImplementedException();
        }

        Option<DjfsResource> sourceO = filesystem.find(DjfsPrincipal.cons(uid), path, Option.of(ReadPreference.primary()));
        if (!sourceO.isPresent()) {
            throw new ResourceNotFoundException(path);
        }
        Operation operation = TrashAppendOperation.create(uid, path, sourceO.get(), callback.getOrElse(""),
                userDiskVersion);
        operationDao.insert(operation);

        CeleryJob celeryJob = CeleryJobUtils.create(TrashAppendOperation.TASK_ID, operation.getUid(),
                operation.getId());
        celeryTaskManager.submit(celeryJob);
        xivaPushGenerator.sendOperationCreatedPush(uid, operation);

        // TODO: fail operation simultaneously in case of submitting failes
        // TODO: if simultaneous fail failed as well -> submit async task to fail operation

        return new AsyncOperationResultPojo(operation, userDiskVersion);
    }

    @WithThreadLocalCache
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    @Path(value = "/legacy/json/bulk_info_by_resource_ids", methods = {HttpMethod.POST})
    @TranslateExceptionToLegacy
    public JsonStringResult bulk_info_by_resource_ids(
            @BoundJsonListByBender ListF<String> rawResourceIds,
            @RequestParam("uid") String rawUid,
            @RequestParam("previewSize") Option<String> customPreviewSize,
            @RequestParam("previewCrop") Option<String> customPreviewCrop,
            @RequestParam("previewType") Option<String> customPreviewType,
            @RequestParam("previewAnimate") Option<String> customPreviewAnimate,
            @RequestParam("previewQuality") Option<String> previewQuality,
            @RequestParam("previewAllowBigSize") Option<String> previewAllowBigSize,
            @RequestParam("meta") Option<String> metaField,
            @RequestListParam("enableServiceIds") Option<SetF<String>> enableServiceIds
    )
    {
        Option<SetF<String>> metaFields = metaField.map(this::parseMetaField);
        PreviewFormattingOptions previewOptions = new PreviewFormattingOptions(customPreviewSize, customPreviewCrop,
                previewQuality, previewAllowBigSize, customPreviewType, customPreviewAnimate);
        TlCache.flush();
        DjfsUid uid = DjfsUid.cons(rawUid, ActionContext.CLIENT_INPUT);
        UserData user = userDao.findExistingAndNotBlocked(uid);

        if (!user.isPg()) {
            throw new DjfsNotImplementedException();
        }

        ListF<DjfsResourceId> resourceIds = rawResourceIds.map(DjfsResourceId::cons);

        ListF<DjfsResourceArea> areasToSearch;
        if (enableServiceIds.isPresent() && enableServiceIds.get().isNotEmpty()) {
            areasToSearch = Cf.arrayList();
            for (String legacyServiceId : enableServiceIds.get()) {
                DjfsResourceArea area;
                try {
                    area = DjfsResourceArea.fromLegacyServiceId(legacyServiceId);
                } catch (InvalidDjfsResourceAreaException e) {
                    throw new LegacyServiceNotFoundException(e);
                }
                if (bulkInfoAllowedAreas.containsTs(area)) {
                    areasToSearch.add(area);
                } else {
                    throw new LegacyServiceNotFoundException(
                            new InvalidDjfsResourceAreaException(legacyServiceId, uid));
                }
            }
        } else {
            areasToSearch = Cf.list(DjfsResourceArea.DISK, DjfsResourceArea.ATTACH, DjfsResourceArea.TRASH,
                    DjfsResourceArea.PHOTOUNLIM);
        }

        DjfsPrincipal principal = DjfsPrincipal.cons(user);
        ListF<DjfsResource> resources = filesystem.find(principal, resourceIds, areasToSearch, Option.of(ReadPreference.secondaryPreferred()));
        resources = filesystem.filterBlocked(resources);

        FormattingContext context = new FormattingContext(user, metaFields, previewOptions, BULK_INFO_BY_RESOURCE_IDS,
                Option.empty(), uid);
        MapF<String, ResourcePojo> resourceIdToResourceMap = resources
                .toMapMappingToKey(r -> r.getResourceId().get().toString())
                .mapValues(resource -> resourcePojoBuilder.build(resource, context));

        ListF<ResourcePojo> resultItems = Cf.arrayList();
        // return resources in same order as in request
        for (DjfsResourceId resourceId : resourceIds) {
            Option<ResourcePojo> resourceO = resourceIdToResourceMap.getO(resourceId.toString());
            resourceO.ifPresent(resultItems::add);
        }

        return new JsonStringResult(new String(resourcePojoSerializer.serializeListJson(resultItems)));
    }

    @WithThreadLocalCache
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    @Path(value = "/legacy/json/bulk_info", methods = {HttpMethod.POST})
    @TranslateExceptionToLegacy
    public JsonStringResult bulk_info(
            @BoundJsonListByBender ListF<String> rawResourcePaths,
            @RequestParam("uid") String rawUid,
            @RequestParam("previewSize") Option<String> customPreviewSize,
            @RequestParam("previewCrop") Option<String> customPreviewCrop,
            @RequestParam("previewType") Option<String> customPreviewType,
            @RequestParam("previewAnimate") Option<String> customPreviewAnimate,
            @RequestParam("previewQuality") Option<String> previewQuality,
            @RequestParam("previewAllowBigSize") Option<String> previewAllowBigSize,
            @RequestParam("meta") Option<String> metaField
    )
    {
        Option<SetF<String>> metaFields = metaField.map(this::parseMetaField);
        PreviewFormattingOptions previewOptions = new PreviewFormattingOptions(customPreviewSize, customPreviewCrop,
                previewQuality, previewAllowBigSize, customPreviewType, customPreviewAnimate);
        TlCache.flush();
        DjfsUid uid = DjfsUid.cons(rawUid, ActionContext.CLIENT_INPUT);
        UserData user = userDao.findExistingAndNotBlocked(uid);

        if (!user.isPg()) {
            throw new DjfsNotImplementedException();
        }

        ListF<DjfsResourcePath> paths = rawResourcePaths.filter(clientInputDataProcessor::isValidPath)
                .map(rawPath -> getPath(rawPath, uid)).filter(Option::isPresent)
                .map(Option::get);

        DjfsPrincipal principal = DjfsPrincipal.cons(user);
        ListF<DjfsResource> resources = filesystem.findByPaths(principal, paths, Option.of(ReadPreference.secondaryPreferred()));

        MapF<String, DjfsResource> pathToResourceMap = resources
                .toMapMappingToKey(resource -> resource.getPath().getArea().getVisiblePathForResource(resource, uid, shareInfoManager, Option.of(ReadPreference.secondaryPreferred())).toString());

        // return resources in same order as in request
        ListF<ResourcePojo> resultItems = Cf.arrayList();
        FormattingContext context = new FormattingContext(user, metaFields, previewOptions, ReadResourceEndpoint.BULK_INFO,
                Option.empty(), uid);
        for (DjfsResourcePath path : paths) {
            Option<DjfsResource> resourceO = pathToResourceMap.getO(path.toString());
            if (!resourceO.isPresent()) {
                continue;
            }
            DjfsResource resource = resourceO.get();
            resultItems.add(resourcePojoBuilder.build(resource, context));
        }

        return new JsonStringResult(new String(resourcePojoSerializer.serializeListJson(resultItems)));
    }

    @Path("/legacy/json/info")
    @TranslateExceptionToLegacy
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    @WithThreadLocalCache
    public JsonStringResult info(@RequestParam("uid") String uidValue,
            @RequestParam("path") String path,
            @RequestParam("meta") Option<String> metaO,
            @RequestParam("tld") Option<String> tld,
            @RequestParam("previewSize") Option<String> customPreviewSize,
            @RequestParam("previewCrop") Option<String> customPreviewCrop,
            @RequestParam("previewType") Option<String> customPreviewType,
            @RequestParam("previewAnimate") Option<String> customPreviewAnimate,
            @RequestParam("previewQuality") Option<String> previewQuality,
            @RequestParam("previewAllowBigSize") Option<String> previewAllowBigSize)
    {
        TlCache.flush();
        Option<SetF<String>> metaFields = metaO.map(this::parseMetaField);
        clientInputDataProcessor.validPathOrThrow(path);
        DjfsUid uid = clientInputDataProcessor.getDjfsUidFromClientInput(uidValue, path);
        UserData user = userDao.findExistingAndNotBlocked(uid);
        DjfsResourcePath resourcePath = getPath(path, uid).getOrThrow(() -> new ResourceNotFoundException(path));
        DjfsResource resource = filesystem.find(DjfsPrincipal.cons(user), resourcePath, Option.of(ReadPreference.secondaryPreferred()))
                .getOrThrow(() -> new ResourceNotFoundException(resourcePath));
        DjfsResourcePath visiblePath = resource.getPath().getArea()
                .getVisiblePathForResource(resource, uid, shareInfoManager, Option.of(ReadPreference.secondaryPreferred()));
        if (!resourcePath.toString().equals(visiblePath.toString()) &&
                !isValidSharePath(uid, resourcePath, visiblePath)) {
            throw new ResourceNotFoundException(resourcePath);
        }
        PreviewFormattingOptions previewOptions = new PreviewFormattingOptions(customPreviewSize, customPreviewCrop,
                previewQuality, previewAllowBigSize, customPreviewType, customPreviewAnimate);
        return new JsonStringResult(new String(resourcePojoSerializer.serializeJson(
                resourcePojoBuilder.build(resource, new FormattingContext(user, metaFields, previewOptions, ReadResourceEndpoint.INFO,
                        Option.of(tld.filter(VALID_TLDS::containsTs).getOrElse("ru")), uid))
        )));
    }

    private boolean isValidSharePath(DjfsUid uid, DjfsResourcePath pathFromInput, DjfsResourcePath discoveredVisiblePath) {
        return uid.asLong() == DjfsUid.SHARE_PRODUCTION.asLong() &&
                clientInputDataProcessor.isShare(pathFromInput.getPath()) &&
                clientInputDataProcessor.isShare(discoveredVisiblePath.getPath()) &&
                pathFromInput.getPath().equals(discoveredVisiblePath.getPath());
    }

    private SetF<String> parseMetaField(String metaField) {
        return Cf.x(metaField.split(",")).filterNot(String::isEmpty).unique();
    }

    private Option<DjfsResourcePath> getPath(String rawPath, DjfsUid uid) {
        if (DjfsResourcePath.containsUid(rawPath)) {
            try {
                DjfsResourcePath path = DjfsResourcePath.cons(rawPath);
                if (path.getUid().equals(uid)) {  // MPFS not return shared resources to participant by owner paths
                    return Option.of(path);
                }
            } catch (InvalidDjfsResourcePathException e) {
                return Option.empty();
            }
        }
        try {
            return Option.of(DjfsResourcePath.cons(uid, rawPath));
        } catch (InvalidDjfsResourcePathException e) {
            return Option.empty();
        }
    }

    @WithThreadLocalCache
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Path(value = "/legacy/json/public_list", methods = {HttpMethod.GET})
    @TranslateExceptionToLegacy
    public JsonStringResult public_list(
            @RequestParam("uid") Option<String> accessUidO,
            @RequestParam("privateHash") String hash,
            @RequestParam("meta") Option<String> metaField,
            @RequestParam("offset") Option<Integer> offsetO,
            @RequestParam("amount") int amount
    )
    {
        final String accessUid = accessUidO.getOrElse("0");
        final UserData accessUser = toUserData(accessUid);

        final int offset = offsetO.getOrElse(0);
        final Option<SetF<String>> metaFields = metaField.map(this::parseMetaField);
        TlCache.flush();
        final PublicAddress publicAddress = publicationManager.getPublicAddress(hash);
        final DjfsResource resource = publicationManager.getPublicResource(accessUid, publicAddress);
        List<DjfsResource> children =
                filesystem.getChildren(DjfsPrincipal.cons(resource.getUid()), resource, offset, amount, Option.of(ReadPreference.secondaryPreferred()));
        final SetF<DjfsResource> cleanedChildren = Cf.hashSet();
        final int pageBlockedItemsNum = publicationManager.countBlockedAndInfectedChildren(children, cleanedChildren);


        final DjfsResource resourceForBlockings = publicationManager.updatePathAndShortUrl(publicAddress, resource)
                .pageBlockedItemsNum(pageBlockedItemsNum).build();
        DjfsResource resourceWithBlockings = publicationManager.checkBlockings(Cf.list(resourceForBlockings)).first();
        if (isBlockingsCheckByHashEnabled) {
            resourceWithBlockings =
                    publicationManager.checkBlockingsByHash(resourceWithBlockings, publicAddress.getEncodedHash());
        }
        final ListF<DjfsResource> childrenWithBlockings = publicationManager
                .processChildren(children, cleanedChildren, pageBlockedItemsNum, resourceWithBlockings);


        FormattingContext context = new FormattingContext(
                accessUser, metaFields, PreviewFormattingOptions.EMPTY, ReadResourceEndpoint.PUBLIC_LIST, Option.empty(),
                accessUser.getId());
        final ResourcePojo result = resourcePojoBuilder.build(resourceWithBlockings, context);
        ListF<ResourcePojo> resultItems = Cf.arrayList(result);
        resultItems.addAll(childrenWithBlockings.map(c -> resourcePojoBuilder.build(c, context)));
        return new JsonStringResult(new String(resourcePojoSerializer.serializeListJson(resultItems)));
    }

    @WithThreadLocalCache
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Path(value = "/legacy/json/public_info", methods = {HttpMethod.GET})
    @TranslateExceptionToLegacy
    public JsonStringResult public_info(
            @RequestParam("uid") Option<String> accessUidO,
            @RequestParam("privateHash") String hash,
            @RequestParam("meta") Option<String> metaField
    )
    {
        final String accessUid = accessUidO.getOrElse("0");
        final UserData accessUser = toUserData(accessUid);

        Option<SetF<String>> metaFields = metaField.map(this::parseMetaField);
        TlCache.flush();
        PublicAddress publicAddress = publicationManager.getPublicAddress(hash);
        DjfsResource resource = null;
        try {
            resource = publicationManager.getPublicResource(accessUid, publicAddress);
        } catch (ResourceNotFoundException ex) {
            publicationManager.fillErrorData(ex, hash);
        }
        DjfsResource resourceForBlockings = publicationManager.updatePathAndShortUrl(publicAddress, resource).build();
        if (!publicAddress.getUser().getYateamUid().isPresent()) {
            resourceForBlockings = publicationManager.checkBlockings(Cf.list(resourceForBlockings)).first();
            if (isBlockingsCheckByHashEnabled) {
                resourceForBlockings =
                        publicationManager.checkBlockingsByHash(resourceForBlockings, publicAddress.getEncodedHash());
            }
        }

        FormattingContext context = new FormattingContext(
                accessUser, metaFields, PreviewFormattingOptions.EMPTY, ReadResourceEndpoint.PUBLIC_INFO, Option.empty(),
                accessUser.getId());
        final ResourcePojo resourcePojo = resourcePojoBuilder.build(resourceForBlockings, context);
        final UserPojo userPojo = publicationManager.constructUserPojo(publicAddress.getUser());
        final PublicInfoPojo publicInfoPojo = new PublicInfoPojo(resourcePojo, userPojo);
        return new JsonStringResult(new String(publicPojoSerializer.serializeJson(publicInfoPojo)));
    }

    @WithThreadLocalCache
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Path(value = "/legacy/json/default_folders", methods = {HttpMethod.GET})
    @TranslateExceptionToLegacy
    public JsonStringResult default_folders(
            @RequestParam("uid") String accessUid,
            @RequestParam("checkExist") Option<String> checkExist
    ) throws JsonProcessingException {
        DjfsUid uid = DjfsUid.cons(accessUid);
        final UserData accessUser = userDao.find(uid)
                .getOrThrow(() -> new UserNotInitializedException(uid));

        UserLocale locale = accessUser.getLocale().getOrElse(UserLocale.DEFAULT_LOCALE);

        MapF<String, String> folders = Cf.list(DefaultFolder.values())
                .toMap(DefaultFolder::value, f -> f.pathForLocale(locale));

        if (checkExist.getOrElse("0").equals("0")) {
            return new JsonStringResult(new String(mapper.writeValueAsBytes(new DefaultFoldersOnlyPathPojo(folders))));
        } else {
            SetF<String> existed = filesystem
                    .findByPaths(DjfsPrincipal.cons(uid), folders.mapEntries((k, f) -> DjfsResourcePath.cons(uid, f)), Option.of(ReadPreference.secondaryPreferred()))
                    .map(DjfsResource::getPath)
                    .map(DjfsResourcePath::getPath)
                    .map(LegacyFilesystemActions::removeLastSlash)
                    .unique();
            MapF<String, DefaultFolderPojo> result = folders
                    .mapValues(path -> {
                        String pathWithoutSlash = removeLastSlash(path);
                        return new DefaultFolderPojo(pathWithoutSlash, existed.containsTs(pathWithoutSlash) ? 1 : 0);
                    });
            return new JsonStringResult(new String(mapper.writeValueAsBytes(new DefaultFoldersFullPojo(result))));
        }
    }

    public static String removeLastSlash(String path) {
        if (path.endsWith("/")) {
            return path.substring(0, path.length() - 1);
        } else {
            return path;
        }
    }

    @NotNull
    public UserData toUserData(String accessUid) {
        try {
            if ("0".equals(accessUid)) {
                return UserData.cons(DjfsUid.cons(accessUid));
            }
            return userDao.find(DjfsUid.cons(accessUid)).getOrElse(() -> UserData.cons(DjfsUid.cons(accessUid)));
        } catch (UserNotInitializedException e) {
            return UserData.cons(DjfsUid.cons(accessUid));
        }
    }
}
