package ru.yandex.chemodan.app.djfs.core.legacy.formatting;

import java.net.URI;
import java.util.function.Supplier;

import com.google.common.escape.Escaper;
import com.google.common.net.PercentEscaper;
import com.mongodb.ReadPreference;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.DownloadUrlAdditionalField;
import ru.yandex.chemodan.app.djfs.core.DownloadUrlGenerator;
import ru.yandex.chemodan.app.djfs.core.client.LogReaderHttpClient;
import ru.yandex.chemodan.app.djfs.core.filesystem.Filesystem;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.AntiVirusScanStatus;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.MediaType;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.PhotosliceAlbumType;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.VersioningStatus;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.mediatype.MediaTypeUtils;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.office.OnlineOfficeFileEditor;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.office.OnlineOfficeFileEditorManager;
import ru.yandex.chemodan.app.djfs.core.publication.PublicationManager;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfo;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfoManager;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.util.MemoizedSupplier;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonValue;
import ru.yandex.inside.passport.blackbox2.Blackbox2;


public class FileMetaProvider extends ResourceMetaProvider {

    public static final String ONLINE_OFFICE_URL = "office_online_url";

    public static final String OFFICE_ONLINE_EDITOR_TYPE = "office_online_editor_type";

    private static final String DEFAULT_CONTENT_TYPE = "application/octet-stream";

    private static final Escaper documentIdUrlEscaper = new PercentEscaper("_.-", false);

    private final DownloadUrlGenerator downloadUrlGenerator;

    private final OnlineOfficeFileEditorManager onlineOfficeFileEditorManager;

    private final ShareInfoManager shareInfoManager;

    private final DynamicProperty<String> onlineOfficeUrlTemplate = new DynamicProperty<>("disk-djfs-online-office-url-template",
            "https://disk.yandex.%s/edit/%s/%s");

    public FileMetaProvider(Blackbox2 blackbox, DownloadUrlGenerator downloadUrlGenerator,
            LogReaderHttpClient logReaderHttpClient, Filesystem filesystem, PublicationManager publicationManager,
            OnlineOfficeFileEditorManager onlineOfficeFileEditorManager, ShareInfoManager shareInfoManager)
    {
        super(filesystem, blackbox, logReaderHttpClient, publicationManager);
        this.downloadUrlGenerator = downloadUrlGenerator;
        this.onlineOfficeFileEditorManager = onlineOfficeFileEditorManager;
        this.shareInfoManager = shareInfoManager;
    }


    Option<URI> getFileUrl(FileDjfsResource file, Supplier<Option<ShareInfo>> shareInfoProvider, FormattingContext context) {
        return getMetaField(context, "file_url",
                () -> Option.of(getFileUrl(context.getUid(), shareInfoProvider, file)));
    }

    Option<URI> getDigestUrl(FileDjfsResource file, Supplier<Option<ShareInfo>> shareInfoProvider, FormattingContext context) {
        return getMetaField(context, "digest_url",
                () -> Option.of(getDigestUrl(context.getUid(), shareInfoProvider, file)));
    }

    Option<AntiVirusScanStatus> getDrweb(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "drweb", file::getAntiVirusScanStatus);
    }

    Option<Long> getEtime(FileDjfsResource file) {
        //compatibility with mpfs answer format. etime (if present) is returned in meta even if not requested
        return file.getEtime();
    }

    Option<MediaType> getMediaType(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "mediatype", () -> getMediaType(file));
    }

    Option<MediaType> getMedia_type(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "media_type", () -> getMediaType(file));
    }

    Option<String> getMimeType(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "mimetype", () -> file.getMimetype().orElse(Option.of(DEFAULT_CONTENT_TYPE)));
    }

    Option<Long> getSize(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "size", () -> Option.of(file.getSize()));
    }

    Option<Instant> getPhotosliceTime(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "photoslice_time", file::getPhotosliceTime);
    }

    Option<String> getStorageType(FileDjfsResource file, FormattingContext context) {
        DjfsResourceArea fileAreaPath = file.getPath().getArea();
        return getMetaField(context, "storage_type",
                () -> Option.when(fileAreaPath.equals(DjfsResourceArea.PHOTOUNLIM), fileAreaPath.rootFolderName));
    }

    Option<PhotosliceAlbumType> getPhotosliceAlbumType(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "photoslice_album_type", file::getPhotosliceAlbumType);
    }

    Option<VersioningStatus> getVersioningStatus(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "versioning_status", () -> Option.of(file.getVersioningStatus()));
    }

    Option<JsonValue> getVideoInfo(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "video_info", () -> file.getVideoInfo().map(JsonObject::parseObject));
    }

    Option<String> getFotkiDataStid(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "fotki_data_stid", () -> file.getExternalProperties().getO("fotki_data_stid"));
    }

    Option<URI> getFotkiDataUrl(FileDjfsResource file, Supplier<Option<ShareInfo>> shareInfoProvider, FormattingContext context) {
        if (!isValidMeta(context, "fotki_data_url")) {
            return Option.empty();
        }

        Option<String> fotkiDataStid = file.getExternalProperties().getO("fotki_data_stid");
        if (fotkiDataStid.isPresent()) {
            return Option.of(downloadUrlGenerator.makeDownloadUrl(
                    context.getUid(),
                    MemoizedSupplier.memoize(() -> getOwnerUid(file, shareInfoProvider.get())),
                    fotkiDataStid.get(),
                    Option.of("text/json"),
                    file.getPath().getName(),
                    true,
                    false
            ));
        }
        return Option.empty();
    }

    Option<String> getNoteName(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "note_name", () -> file.getExternalProperties().getO("note_name"));
    }

    Option<String> getNoteRevisionCreated(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "note_revision_created",
                () -> file.getExternalProperties().getO("note_revision_created"));
    }

    Option<String> getNoteRevisionDeleted(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "note_revision_deleted",
                () -> file.getExternalProperties().getO("note_revision_deleted"));
    }

    Option<String> getFileMid(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "file_mid", () -> Option.of(file.getFileStid()));
    }

    Option<String> getMd5(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "md5", () -> Option.of(file.getMd5()));
    }

    Option<String> getSha256(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "sha256", () -> Option.of(file.getSha256()));
    }

    Option<Double> getAesthetics(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "aesthetics", file::getAesthetics);
    }

    Option<Integer> getWidth(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "width", file::getWidth);
    }

    Option<Integer> getHeight(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "height", file::getHeight);
    }

    Option<Integer> getAngle(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "angle", file::getAngle);
    }

    Option<ListF<String>> getAlbumsExclusions(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "albums_exclusions", file::getAlbumsExclusions);
    }

    Option<String> getPmid(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "pmid", file::getPreviewStid);
    }

    Option<MapF<MetaPreviewUrlSize, URI>> getPreviewSizes(
            FileDjfsResource file, Supplier<Option<ShareInfo>> shareInfoProvider, FormattingContext context)
    {
        return getMetaField(context, "sizes", () -> getPreviewUrlsBySize(file, shareInfoProvider, context));
    }

    Option<URI> getCustomPreview(FileDjfsResource file, Supplier<Option<ShareInfo>> shareInfoProvider, FormattingContext context) {
        if (!isValidMeta(context, "custom_preview")) {
            return Option.empty();
        }

        Option<MapF<MetaPreviewUrlSize, URI>> previewsO = getPreviewUrlsBySize(file, shareInfoProvider, context);
        if (previewsO.isEmpty()) {
            return Option.empty();
        }

        MapF<MetaPreviewUrlSize, URI> previews = previewsO.get();
        //default value - S preview
        return previews.containsKeyTs(MetaPreviewUrlSize.C)
                ? previews.getO(MetaPreviewUrlSize.C) : previews.getO(MetaPreviewUrlSize.S);
    }

    public Option<FileMetaPojo.OnlineOfficeData> getOnlineOfficeData(FileDjfsResource file, FormattingContext context) {
        return getMetaFields(context, Cf.list(ONLINE_OFFICE_URL, OFFICE_ONLINE_EDITOR_TYPE),
                () -> getOnlineOfficeDataField(file, context), FileMetaPojo.OnlineOfficeData::fromMetaProviderMap);
    }

    public Option<String> getDigestMid(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "digest_mid", () -> Option.of(file.getDigestStid()));
    }

    public Option<String> getHid(FileDjfsResource file, FormattingContext context) {
        return getMetaField(context, "hid", () -> Option.of(file.getHid()));
    }

    private MapF<String, String> getOnlineOfficeDataField(FileDjfsResource file, FormattingContext context) {
        if (!context.getTld().isPresent()) {
            return Cf.map();
        }
        Option<ReadPreference> readPreference = Option.of(context.getReadResourceEndpoint().getReadPreference());
        DjfsResourcePath visiblePath = file.getPath().getArea()
                .getVisiblePathForResource(file, context.getUid(), shareInfoManager, readPreference);
        return onlineOfficeFileEditorManager.getEditorForUser(context.getRequestUser())
                .filter(editor -> editor.isEditPossible(file, visiblePath))
                .map(editor -> buildOnlineOfficeData(file, context, editor, Option.of(visiblePath), context.getTld().map(String::trim).get()))
                .getOrElse(Cf::map);
    }

    private MapF<String, String> buildOnlineOfficeData(FileDjfsResource file, FormattingContext context,
                                                       OnlineOfficeFileEditor editor, Option<DjfsResourcePath> visiblePath, String tld) {
        OnlineOfficeClientType clientType = context.getReadResourceEndpoint().getOnlineOfficeClientType();
        return clientType.getDocumentId(file, visiblePath).map(documentIdUrlEscaper::escape).map(documentId ->
                Cf.map(
                        ONLINE_OFFICE_URL, String.format(onlineOfficeUrlTemplate.get(),
                                tld,
                                clientType.name().toLowerCase(),
                                documentId),
                        OFFICE_ONLINE_EDITOR_TYPE, editor.getEditorType()
                )).getOrElse(Cf::map);
    }

    private Option<MapF<MetaPreviewUrlSize, URI>> getPreviewUrlsBySize(FileDjfsResource file,
            Supplier<Option<ShareInfo>> shareInfoSupplier, FormattingContext context)
    {
        if (!file.getPreviewStid().isPresent()) {
            return Option.empty();
        }

        MapF<MetaPreviewUrlSize, URI> sizes = Cf.hashMap();
        for (MetaPreviewUrlSize size : MetaPreviewUrlSize.values()) {
            Option<URI> uri = size.getPreviewUrlBuilder().buildUrl(
                    downloadUrlGenerator, size, context.getUid(), MemoizedSupplier.memoize(() -> getOwnerUid(file, shareInfoSupplier.get())), file,
                    context.getPreviewOption(), context.getReadResourceEndpoint().isEternalPreview() && !context.getUid().isShared());

            uri.ifPresent(u -> sizes.put(size, u));
        }

        return Option.of(sizes);
    }


    private Option<MediaType> getMediaType(FileDjfsResource file) {
        return MediaTypeUtils.getFileMediaTypeO(file).orElse(Option.of(MediaType.UNKNOWN));
    }

    private URI getFileUrl(DjfsUid requestUid, Supplier<Option<ShareInfo>> shareInfoSupplier, FileDjfsResource file) {
        return downloadUrlGenerator.prepareDownloadUrl(
                requestUid, MemoizedSupplier.memoize(() -> getOwnerUid(file, shareInfoSupplier.get())), file, false, false, FileDjfsResource::getFileStid,
                FileDjfsResource::getMimetype, DownloadUrlAdditionalField.ALL)
                .build();
    }

    private URI getDigestUrl(DjfsUid requestUid, Supplier<Option<ShareInfo>> shareInfoSupplier, FileDjfsResource file) {
        return downloadUrlGenerator.prepareDownloadUrl(
                requestUid, MemoizedSupplier.memoize(() -> getOwnerUid(file, shareInfoSupplier.get())), file, false, false, FileDjfsResource::getDigestStid,
                fileDjfsResource -> Option.empty(), Cf.set())
                .build();
    }
}
