package ru.yandex.chemodan.app.djfs.core.legacy.formatting;

import java.net.URI;
import java.util.function.Supplier;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.djfs.core.DownloadUrlAdditionalField;
import ru.yandex.chemodan.app.djfs.core.DownloadUrlGenerator;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.MediaType;
import ru.yandex.chemodan.app.djfs.core.legacy.PreviewFormattingOptions;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.misc.enums.StringEnum;
import ru.yandex.misc.io.http.UriBuilder;

public enum MetaPreviewUrlSize implements StringEnum {
    ORIGINAL(new OriginalUrlBuilder()),
    DEFAULT(new DefaultUrlBuilder()),
    XXXS(new SimpleURLBuilder()),
    XXS(new SimpleURLBuilder()),
    XS(new SimpleURLBuilder()),
    S(new SimpleURLBuilder()),
    M(new SimpleURLBuilder()),
    L(new SimpleURLBuilder()),
    XL(new SimpleURLBuilder()),
    XXL(new SimpleURLBuilder()),
    XXXL(new SimpleURLBuilder()),
    C(new CustomUrlBuilder());

    private final URLBuilder previewUrlBuilder;

    MetaPreviewUrlSize(URLBuilder previewUrlBuilder) {
        this.previewUrlBuilder = previewUrlBuilder;
    }

    public URLBuilder getPreviewUrlBuilder() {
        return previewUrlBuilder;
    }

    @Override
    public String value() {
        return name().toUpperCase();
    }


    public interface URLBuilder {
        Option<URI> buildUrl(DownloadUrlGenerator downloadUrlGenerator, MetaPreviewUrlSize size, DjfsUid requestUid,
                             Supplier<DjfsUid> ownerUidProvider, FileDjfsResource file, PreviewFormattingOptions previewOptions, boolean isEternal);
    }

    private static class SimpleURLBuilder implements URLBuilder {
        @Override
        public Option<URI> buildUrl(DownloadUrlGenerator downloadUrlGenerator, MetaPreviewUrlSize size,
                DjfsUid requestUid,
                Supplier<DjfsUid> ownerUidProvider, FileDjfsResource file, PreviewFormattingOptions previewOptions, boolean isEternal)
        {
            UriBuilder uri = downloadUrlGenerator.prepareUserPreviewUrl(
                    requestUid, ownerUidProvider, file, Option.of(size.value()), Option.of("0"), true, isEternal);
            return Option.of(addQuality(uri, previewOptions).build());
        }
    }

    private static class OriginalUrlBuilder implements URLBuilder {
        private static final SetF<String> allowedMimeTypesForInlining = Cf.set(
                "image/jpg",
                "image/jpeg",
                "image/gif",
                "image/apng",
                "image/png",
                "image/bmp",
                "image/ico",
                "image/webp",
                "image/tiff"
        );

        @Override
        public Option<URI> buildUrl(DownloadUrlGenerator downloadUrlGenerator, MetaPreviewUrlSize size,
                DjfsUid requestUid,
                Supplier<DjfsUid> ownerUidProvider, FileDjfsResource file, PreviewFormattingOptions previewOptions, boolean isEternal)
        {
            // ссылка на оригинал бывает только у изображений
            if (!file.getMediaType().filter(mt -> mt.equals(MediaType.IMAGE)).isPresent()) {
                return Option.empty();
            }
            boolean inline = file.getMimetype().map(allowedMimeTypesForInlining::containsTs).getOrElse(false);
            UriBuilder uri = downloadUrlGenerator.prepareDownloadUrl(requestUid, ownerUidProvider, file, inline, false,
                    FileDjfsResource::getFileStid, FileDjfsResource::getMimetype, DownloadUrlAdditionalField.ALL);
            return Option.of(addQuality(uri, previewOptions).build());
        }
    }

    private static class DefaultUrlBuilder implements URLBuilder {
        @Override
        public Option<URI> buildUrl(DownloadUrlGenerator downloadUrlGenerator, MetaPreviewUrlSize size,
                DjfsUid requestUid,
                Supplier<DjfsUid> ownerUidProvider, FileDjfsResource file, PreviewFormattingOptions previewOptions, boolean isEternal)
        {
            UriBuilder uri = downloadUrlGenerator.prepareUserPreviewUrl(requestUid, ownerUidProvider, file, Option.empty(),
                    Option.empty(), true, isEternal);
            if (previewOptions.isAllowBigSize()) {
                uri = uri.addParam("allow_big_size", 1);
            }

            return Option.of(addQuality(uri, previewOptions).build());
        }
    }

    private static class CustomUrlBuilder implements URLBuilder {
        @Override
        public Option<URI> buildUrl(DownloadUrlGenerator downloadUrlGenerator, MetaPreviewUrlSize size,
                DjfsUid requestUid,
                Supplier<DjfsUid> ownerUidProvider, FileDjfsResource file, PreviewFormattingOptions previewOptions, boolean isEternal)
        {
            if (!previewOptions.getCustomPreviewSize().isPresent()) {
                return Option.empty();
            }

            Option<String> crop = previewOptions.getCustomPreviewCrop().isPresent()
                    ? previewOptions.getCustomPreviewCrop() : Option.of("0");

            UriBuilder customPreviewBuilder = downloadUrlGenerator.prepareUserPreviewUrl(
                    requestUid, ownerUidProvider, file, previewOptions.getCustomPreviewSize(), crop, true, isEternal);

            previewOptions.getCustomPreviewType()
                    .ifPresent(type -> customPreviewBuilder.addParam("preview_type", type));
            if (previewOptions.isCustomPreviewAnimate()) {
                customPreviewBuilder.addParam("animate", true);
            }
            if (previewOptions.isAllowBigSize()) {
                customPreviewBuilder.addParam("allow_big_size", 1);
            }

            return Option.of(addQuality(customPreviewBuilder, previewOptions).build());
        }
    }

    private static UriBuilder addQuality(UriBuilder uri, PreviewFormattingOptions options) {
        if (options.getPreviewQuality().isPresent()) {
            return uri.addParam("quality", options.getPreviewQuality().get());
        }
        return uri;
    }
}
