package ru.yandex.chemodan.app.djfs.core.legacy.formatting;

import java.util.function.Supplier;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.djfs.core.client.LogReaderHttpClient;
import ru.yandex.chemodan.app.djfs.core.filesystem.Filesystem;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.publication.PublicationManager;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfo;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonValue;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;


public abstract class ResourceMetaProvider {

    private static final String PATH_SEPARATOR = "/";

    public static final String VERSION_DISK_DIST = "version|urn:yandex:disk:dist";

    public static final String CDN_DISK_DIST = "cdn|urn:yandex:disk:dist:url";

    public static final String WIN_32_ATTRIBUTTES = "Win32FileAttributes|urn:schemas-microsoft-com:";

    public static final String LINK_CLIENT_META = "link|urn:yandex:disk:client:meta";

    protected final Blackbox2 blackbox;
    protected final Filesystem filesystem;
    protected final LogReaderHttpClient logReaderHttpClient;
    protected final PublicationManager publicationManager;

    public ResourceMetaProvider(Filesystem filesystem, Blackbox2 blackbox,
            LogReaderHttpClient logReaderHttpClient, PublicationManager publicationManager)
    {
        this.blackbox = blackbox;
        this.logReaderHttpClient = logReaderHttpClient;
        this.filesystem = filesystem;
        this.publicationManager = publicationManager;
    }

    static <T> Option<T> getMetaFieldOnlyWithIntent(FormattingContext context, String metaField,
            Function0<Option<T>> f)
    {
        if (context.getRequestMetaFields().isPresent() && context.getRequestMetaFields().get().containsTs(metaField)) {
            return f.apply();
        } else {
            return Option.empty();
        }
    }


    static <T> Option<T> getMetaField(FormattingContext context, String metaField, Function0<Option<T>> f) {
        if (isValidMeta(context, metaField)) {
            return f.apply();
        } else {
            return Option.empty();
        }
    }

    public static <T, U> Option<T> getMetaFields(FormattingContext context, ListF<String> metaFields,
            Supplier<MapF<String, U>> dataSupplier, Function<MapF<String, U>, T> mapper)
    {
        if (!isValidAnyOfMeta(context, metaFields)) {
            return Option.empty();
        }
        MapF<String, U> result = dataSupplier.get().filterKeys(field -> isValidMeta(context, field));
        return Option.of(mapper.apply(result));
    }

    static boolean isValidMeta(FormattingContext context, String meta) {
        return context.getRequestMetaFields().isPresent() &&
                (context.getRequestMetaFields().get().containsTs(meta) ||
                        context.getRequestMetaFields().get().isEmpty());
    }

    private static boolean isValidAnyOfMeta(FormattingContext context, ListF<String> metaFields) {
        return metaFields.exists(field -> isValidMeta(context, field));
    }

    public Option<Boolean> isBlocked(DjfsResource resource, FormattingContext context) {
        return getMetaField(context, "blocked", () -> Option.of(resource.isBlocked()));
    }

    public Option<Boolean> isVisible(DjfsResource resource, FormattingContext context) {
        return getMetaField(context, "visible", () -> Option.of(resource.isVisible()));
    }

    public Option<Boolean> isPublic(DjfsResource resource, FormattingContext context) {
        return getMetaField(context, "public", () -> Option.of(resource.isPublic()));
    }

    public Option<DjfsFileId> getFileId(DjfsResource resource, FormattingContext context) {
        return getMetaField(context, "file_id", resource::getFileId);
    }

    public Option<DjfsResourceId> getResourceId(DjfsResource resource, FormattingContext context) {
        return getMetaField(context, "resource_id", resource::getResourceId);
    }

    public Option<JsonValue> getCustomProperties(DjfsResource resource, FormattingContext context) {
        return getMetaField(context, "custom_properties",
                () -> resource.getCustomProperties().map(JsonObject::parseObject));
    }

    public Option<String> getShortUrl(DjfsResource resource, FormattingContext context) {
        if (!isValidMeta(context, "short_url")) {
            return Option.empty();
        }

        if (publicationManager.isYaTeamSubtree(resource) &&
                publicationManager.userHasNdaRights(context.getRequestUser()))
        {
            return resource.getPublicHash()
                    .map(publicationManager::createNdaResourceUrl).orElse(resource.getShortUrl());
        }
        return resource.getShortUrl();
    }

    Option<GroupInfo> getGroupInfo(
            DjfsResource resource, Supplier<Option<ShareInfo>> shareInfoProvider, FormattingContext context)
    {
        if (!isValidMeta(context, "group")) {
            return Option.empty();
        }

        return shareInfoProvider.get().map(s -> buildGroupInfo(context.getRequestUser().getUid(), s, resource));
    }

    public DjfsUid getOwnerUid(DjfsResource resource, Option<ShareInfo> shareInfoO) {
        return shareInfoO.map(ShareInfo::getOwnerUid).getOrElse(resource.getUid());
    }

    public Option<Integer> getDownloadCounter(DjfsResource resource, FormattingContext context) {
        return getMetaField(context, "download_counter", resource::getDownloadCounter);
    }

    Option<Long> getViewsCounter(DjfsResource resource, FormattingContext context) {
        if (!isValidMeta(context, "views_counter")) {
            return Option.empty();
        }
        return resource.getPath().getArea().getResourceAreaMetaProvider()
                .getViewsCounterForResource(resource, logReaderHttpClient);
    }

    Option<Integer> getPageBlockedItemsNum(DjfsResource resource, FormattingContext context) {
        return getMetaField(context, "page_blocked_items_num", resource::getPageBlockedItemsNum);
    }

    Option<CommentId> getCommentIds(DjfsResource resource, FormattingContext context) {
        if (!isValidMeta(context, "comment_ids")) {
            return Option.empty();
        }

        //nobody uses these fields, they are differ from mpfs, they would be deleted
        if (resource.getResourceId().isPresent()) {
            String publicResource = resource.getResourceId().get().getValue();
            String privateResource = resource.getResourceId().get().getValue(); //incorrect in case resource is shared
            return Option.of(new CommentId(publicResource, privateResource));
        }
        return Option.empty();
    }

    Option<Blockings> getBlockings(DjfsResource resource, FormattingContext context) {
        return getMetaField(context, "blockings", resource::getBlockings);
    }

    Option<Long> getRevision(DjfsResource resource, FormattingContext context) {
        return getMetaField(context, "revision", resource::getVersion);
    }

    Option<String> getPublicHash(DjfsResource resource, FormattingContext context) {
        return getMetaField(context, "public_hash", resource::getPublicHash);
    }

    Option<Instant> getTrashAppendTime(DjfsResource resource, FormattingContext context) {
        return getMetaField(context, "append_time",
                () -> attributeForRemovedResource(resource, resource::getTrashAppendTime));
    }

    Option<String> getPathBeforeRemoved(DjfsResource resource, FormattingContext context) {
        return getMetaField(context, "original_id",
                () -> attributeForRemovedResource(resource, resource::getTrashAppendOriginalPath));

    }

    public Option<String> getOriginalParenId(DjfsResource resource, FormattingContext context) {
        return getMetaField(context, "original_parent_id",
                () -> attributeForRemovedResource(resource, resource::getTrashAppendOriginalPath).map(this::getParentPath));
    }

    public Option<String> getFsSymbolicLink(DjfsResource resource, FormattingContext context) {
        return getMetaField(context, "fs_symbolic_link", resource::getFsSymbolycLink);
    }

    public Option<String> getExternalProperty(DjfsResource resource, FormattingContext context, String property) {
        return getMetaField(context, property, () -> resource.getExternalProperties().getO(property));
    }

    private String getParentPath(String path) {
        String[] pathParts = path.split(PATH_SEPARATOR);
        return Cf.x(pathParts).take(pathParts.length - 1).mkString("", PATH_SEPARATOR, PATH_SEPARATOR);
    }

    private static <T> Option<T> attributeForRemovedResource(DjfsResource resource,
            Function0<Option<T>> attributeProvider)
    {
        if (resource.getPath().getArea() == DjfsResourceArea.TRASH) {
            return attributeProvider.apply();
        }

        return Option.empty();
    }


    private GroupInfo buildGroupInfo(DjfsUid requestUid, ShareInfo shareInfo, DjfsResource resource) {
        DjfsUid ownerUid = shareInfo.getOwnerUid();
        BlackboxCorrectResponse userInfo = BlackboxUtils.getBlackboxUserInfo(ownerUid, blackbox);
        GroupOwner groupOwnerPojo = new GroupOwner(userInfo);
        return new GroupInfo(requestUid, shareInfo, resource, groupOwnerPojo);
    }
}
