package ru.yandex.chemodan.app.djfs.core.legacy.formatting.mediatype;

import java.util.HashSet;
import java.util.Set;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.MediaType;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.StringUtils;

import static ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath.EXTENSION_DELIMITER;

/**
 * @author yashunsky
 */
public class MediaTypeUtils {
    private static final String MIME_TYPES_FILE = "/etc/mime.types";
    private static final String FALLBACK_FILE_NAME =
            "ru/yandex/chemodan/app/djfs/core/legacy/formatting/mediatype/fallback_mime.types";

    private static final String DEFAULT_FILE_NAME =
            "ru/yandex/chemodan/app/djfs/core/legacy/formatting/mediatype/default_mime.types";

    private static final ListF<MediaTypeExtensions> MEDIA_TYPE_EXTENSIONS =
            Cf.list(
                    new MediaTypeExtensions(MediaType.SPREADSHEET),
                    new MediaTypeExtensions(MediaType.DEVELOPMENT),
                    new MediaTypeExtensions(MediaType.WEB),
                    new MediaTypeExtensions(MediaType.EXECUTABLE),
                    new MediaTypeExtensions(MediaType.FONT),
                    new MediaTypeExtensions(MediaType.BACKUP),
                    new MediaTypeExtensions(MediaType.SETTINGS),
                    new MediaTypeExtensions(MediaType.TEXT),
                    new MediaTypeExtensions(MediaType.IMAGE),
                    new MediaTypeExtensions(MediaType.FLASH),
                    new MediaTypeExtensions(MediaType.DISKIMAGE),
                    new MediaTypeExtensions(MediaType.BOOK),
                    new MediaTypeExtensions(MediaType.VIDEO),
                    new MediaTypeExtensions(MediaType.COMPRESSED),
                    new MediaTypeExtensions(MediaType.ENCODED),
                    new MediaTypeExtensions(MediaType.DOCUMENT),
                    new MediaTypeExtensions(MediaType.DATA),
                    new MediaTypeExtensions(MediaType.AUDIO),
                    new MediaTypeExtensions(MediaType.UNKNOWN)
            );

    private static final MapF<MediaType, SetF<String>> EXTENSIONS_BY_MEDIA_TYPE = Cf.x(MEDIA_TYPE_EXTENSIONS
            .toTuple2List(MediaTypeExtensions::getMediaType, MediaTypeExtensions::getExtensionsSet)
            .toJavaLinkedHashMap());

    private static final MapF<String, SetF<String>> EXTENSIONS_BY_MIME_TYPE = loadExtensionsByMimeType();

    private static MapF<String, SetF<String>> loadExtensionsByMimeType() {
        File2 mimesType = new File2(MIME_TYPES_FILE);
        ListF<String> lines;
        if (mimesType.exists()) {
            lines = mimesType.readLines();
        } else {
            lines = new ClassPathResourceInputStreamSource(FALLBACK_FILE_NAME).readLines();
        }

        MapF<String, Set<String>> extensionsByMimeType = Cf.hashMap();
        readMimeTypes(new ClassPathResourceInputStreamSource(DEFAULT_FILE_NAME).readLines(), extensionsByMimeType);
        readMimeTypes(lines, extensionsByMimeType);

        return extensionsByMimeType.mapValues(Cf::toSet);
    }

    private static void readMimeTypes(ListF<String> lines, MapF<String, Set<String>> extensionsByMimeType) {
        lines.filterNot(line -> line.startsWith("#")).filterNot(String::isEmpty).forEach(line -> {
            ListF<String> parts = Cf.x(line.split("[ \t]+"));
            if (parts.length() < 2) {
                return;
            }
            String mimeType = parts.first();
            SetF<String> extensions = parts.drop(1).map(String::toLowerCase).unique();

            extensionsByMimeType.getOrElseUpdate(mimeType, HashSet::new).addAll(extensions);
        });
    }

    public static Option<MediaType> getMediaTypeByExtension(String extension) {
        String ext = extension.toLowerCase();
        Option<MediaType> firstCommonMediaType = Option.empty();
        for (MediaTypeExtensions mediaTypeExtension : MEDIA_TYPE_EXTENSIONS) {
            Option<Boolean> foundExtension = mediaTypeExtension.extensions.getO(ext);
            if (!foundExtension.isPresent()) {
                continue;
            }

            if (foundExtension.get()) {
                return Option.of(mediaTypeExtension.mediaType);
            } else if (!firstCommonMediaType.isPresent()) {
                firstCommonMediaType = Option.of(mediaTypeExtension.mediaType);
            }
        }

        return firstCommonMediaType;
    }

    public static Option<MediaType> getMediaTypeByMimeType(String mimeType) {
        SetF<String> extensionCandidates = EXTENSIONS_BY_MIME_TYPE.getO(mimeType).getOrElse(Cf.set());

        int max = 0;
        Option<MediaType> candidate = Option.empty();
        for (Tuple2<MediaType, Integer> tuple2 : EXTENSIONS_BY_MEDIA_TYPE.entries()
                .map2(exts -> extensionCandidates.count(exts::containsTs))) {
            if (tuple2._2 >= max) {
                max = tuple2._2;
                candidate = Option.of(tuple2._1);
            }
        }

        return candidate;
    }

    public static Option<MediaType> getFileMediaTypeO(FileDjfsResource file) {
        // CHEMODAN-14123
        if (file.getVideoInfo().isPresent()) {
            return Option.of(MediaType.VIDEO);
        }

        if (file.getMediaType().isPresent()) {
            return file.getMediaType();
        }

        return MediaTypeUtils.getMediaTypeByExtensionOrMimeType(
                StringUtils.substringAfterLast(file.getDisplayName(), EXTENSION_DELIMITER), file.getMimetype());
    }

    public static Option<MediaType> getMediaTypeByExtensionOrMimeType(String extension, Option<String> mimeTypeO) {
        return getMediaTypeByExtension(extension).orElse(
                () -> mimeTypeO.filterMap(MediaTypeUtils::getMediaTypeByMimeType));
    }

    private static class MediaTypeExtensions {
        private static final String CLASSPATH_PREFIX =
                "ru/yandex/chemodan/app/djfs/core/legacy/formatting/mediatype/extensions_";

        private final MediaType mediaType;
        private final MapF<String, Boolean> extensions;

        MediaTypeExtensions(MediaType mediaType)
        {
            this.mediaType = mediaType;
            ListF<String> lines = new ClassPathResourceInputStreamSource(
                    CLASSPATH_PREFIX + mediaType.getStringRepresentation() + ".properties").readLines();
            this.extensions = lines.filterNot(line -> line.startsWith("#")).filterNot(String::isEmpty)
                    .toMap(
                            s -> StringUtils.substringBefore(s, "=").toLowerCase(),
                            s -> "1".equals(StringUtils.substringAfter(s, "=")));
        }

        MediaType getMediaType() {
            return mediaType;
        }

        SetF<String> getExtensionsSet() {
            return extensions.keySet();
        }
    }

}
