package ru.yandex.chemodan.app.djfs.core.legacy.web;

import lombok.RequiredArgsConstructor;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.djfs.core.DjfsException;
import ru.yandex.chemodan.app.djfs.core.album.AlbumUnableToAppendException;
import ru.yandex.chemodan.app.djfs.core.album.AlbumUnableToDeleteException;
import ru.yandex.chemodan.app.djfs.core.album.AlbumsNotFoundException;
import ru.yandex.chemodan.app.djfs.core.filesystem.InvalidNewFolderNameException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.ForbiddenResourcePathException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.NoParentFolderException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.NoPermissionException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.OfficeFileTooLargeException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.OperationInAreaNotPermittedException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.ParentIsFileException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.PrincipalBlockedException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.ResourceBlockedException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.ResourceExistsException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.ResourceNotFoundException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.SameSourceAndDestinationException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.SourceIsParentForDestinationException;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.exception.InvalidClientInputDjfsPublicHashException;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.exception.InvalidDjfsResourcePathException;
import ru.yandex.chemodan.app.djfs.core.filesystem.operation.postprocess.PostProcessTaskLimitExceededException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyBadRequestException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyDecryptionErrorException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyGroupNoPermissionException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyMpfsException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyMpfsExceptionOrigin;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyOfficeTooLargeException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyPathErrorException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyResourceAlreadyExists;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyResourceBlockedException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyResourceLockedException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyResourceNotFoundException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyStorageAddressError;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyTooManyExecutingOperationException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyUserBlockedException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyUserIsReadOnlyException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyWhUserNotInitializedException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.albums.LegacyAlbumNotFoundException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.albums.LegacyAlbumUnableToAppendException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.albums.LegacyAlbumUnableToDeleteException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.mkdir.LegacyMkdirNoParentFolderException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.mkdir.LegacyMkdirPermissionDeniedException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.mkdir.LegacyMkdirResourceExistsException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.move.LegacyMoveNoTargetParentException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.move.LegacyMoveParentToChildException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.move.LegacyMoveSameDestinationException;
import ru.yandex.chemodan.app.djfs.core.lock.FilesystemLock;
import ru.yandex.chemodan.app.djfs.core.lock.LockManager;
import ru.yandex.chemodan.app.djfs.core.lock.ResourceLockedException;
import ru.yandex.chemodan.app.djfs.core.lock.UserLockedException;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfoManager;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.PathIsInvalidException;
import ru.yandex.chemodan.app.djfs.core.user.UserIsBlockedException;
import ru.yandex.chemodan.app.djfs.core.user.UserNotInitializedException;
import ru.yandex.commune.a3.action.ActionDescriptor;
import ru.yandex.commune.a3.action.invoke.ActionInvocationContext;
import ru.yandex.commune.a3.action.parameter.WebRequest;
import ru.yandex.commune.a3.action.result.error.ExceptionHandler;
import ru.yandex.commune.a3.action.result.error.ExceptionHandlerOrders;
import ru.yandex.commune.a3.action.result.error.ExceptionResolver;

/**
 * @author eoshch
 */
@RequiredArgsConstructor
public class LegacyMpfsExceptionHandler implements ExceptionHandler {
    private final LockManager lockManager;
    private final ShareInfoManager shareInfoManager;

    @Override
    public Option<?> handleException(WebRequest request, ActionInvocationContext context,
            ExceptionResolver exceptionResolver, Exception exception)
    {
        if (!(exception instanceof DjfsException)) {
            return Option.empty();
        }

        Option<ActionDescriptor> actionDescriptorO = context.getActionDescriptor();
        if (!actionDescriptorO.isPresent()) {
            return Option.empty();
        }

        Option<TranslateExceptionToLegacy> annotationO = actionDescriptorO.get()
                .getAnnotationO(TranslateExceptionToLegacy.class);
        if (!annotationO.isPresent()) {
            return Option.empty();
        }

        Option<DjfsUid> uid = request.getParameter("uid").firstO().filter(DjfsUid::isValid).map(DjfsUid::cons);
        Option<LegacyMpfsException> legacyExceptionO = translate((DjfsException) exception, uid,
                annotationO.get().origin());

        if (!legacyExceptionO.isPresent()) {
            return Option.empty();
        }

        LegacyMpfsException e = legacyExceptionO.get();

        context.getHttpContext().setStatusCode(e.getHttpStatusCode().getStatusCode());
        context.getHttpContext().addHeader("X-MPFS-Legacy-Error-Code", Integer.toString(e.getCode()));

        LegacyMpfsErrorResult result = LegacyMpfsErrorResult.builder()
                .code(e.getCode())
                .title(e.getTitle())
                .data(e.getAdditionalData())
                .build();

        return Option.of(result);

    }

    public Option<LegacyMpfsException> translate(DjfsException exception, DjfsUid uid,
            LegacyMpfsExceptionOrigin origin)
    {
        return translate(exception, Option.of(uid), origin);
    }

    private Option<LegacyMpfsException> translate(DjfsException exception, Option<DjfsUid> uid) {
        if (exception instanceof LegacyMpfsException) {
            return Option.of(exception).cast();
        }

        if (exception instanceof UserNotInitializedException) {
            return Option.of(new LegacyWhUserNotInitializedException(exception));
        }

        if ((exception instanceof PrincipalBlockedException) || (exception instanceof UserIsBlockedException)) {
            return Option.of(new LegacyUserBlockedException(exception));
        }

        if (exception instanceof UserLockedException) {
            return Option.of(new LegacyUserIsReadOnlyException(exception));
        }

        if (exception instanceof ForbiddenResourcePathException) {
            return Option.of(new LegacyStorageAddressError(exception));
        }

        if (exception instanceof ResourceNotFoundException) {
            return Option.of(new LegacyResourceNotFoundException(exception));
        }

        if (exception instanceof ResourceBlockedException) {
            return Option.of(new LegacyResourceBlockedException(exception, ((ResourceBlockedException) exception).data));
        }

        if (exception instanceof ResourceLockedException) {
            ResourceLockedException casted = (ResourceLockedException) exception;
            DjfsResourcePath path = uid
                    .filterMap(x -> shareInfoManager.get(casted.path).map(y -> Tuple2.tuple(y, x)))
                    .filterMap(x -> x._1.ownerPathToParticipantPath(casted.path, x._2))
                    .getOrElse(casted.path);

            Option<FilesystemLock> lock = lockManager.getLocks(casted.path).firstO();
            if (lock.isPresent()) {
                return Option.of(new LegacyResourceLockedException(lock.get().getUid(), path,
                        lock.get().getOperationId(), lock.get().getOperationType(), exception));
            } else {
                return Option.of(new LegacyResourceLockedException(casted.path.getUid(), path, exception));
            }
        }

        if (exception instanceof NoPermissionException) {
            // todo: похоже нам нужно уметь различать нехватку прав в группе от остальных для обратной совместимости
            return Option.of(new LegacyGroupNoPermissionException(exception));
        }

        if (exception instanceof InvalidNewFolderNameException) {
            return Option.of(new LegacyPathErrorException(exception));
        }

        if (exception instanceof InvalidClientInputDjfsPublicHashException) {
            return Option.of(new LegacyDecryptionErrorException(exception));
        }

        if (exception instanceof OfficeFileTooLargeException) {
            return Option.of(new LegacyOfficeTooLargeException(exception));
        }

        if (exception instanceof PathIsInvalidException) {
            return Option.of(new LegacyBadRequestException(exception));
        }

        if (exception instanceof AlbumsNotFoundException) {
            return Option.of(new LegacyAlbumNotFoundException(exception));
        }

        if (exception instanceof AlbumUnableToDeleteException) {
            return Option.of(new LegacyAlbumUnableToDeleteException(exception));
        }

        if (exception instanceof AlbumUnableToAppendException) {
            return Option.of(new LegacyAlbumUnableToAppendException(exception));
        }
        return Option.empty();
    }

    private Option<LegacyMpfsException> translate(DjfsException exception, Option<DjfsUid> uid,
            LegacyMpfsExceptionOrigin origin)
    {
        if (exception instanceof LegacyMpfsException) {
            return Option.of(exception).cast();
        }

        if (origin == LegacyMpfsExceptionOrigin.MKDIR) {
            if (exception instanceof ResourceExistsException) {
                return Option.of(new LegacyMkdirResourceExistsException(exception));
            }

            if ((exception instanceof NoParentFolderException) || (exception instanceof ParentIsFileException)) {
                return Option.of(new LegacyMkdirNoParentFolderException(exception));
            }

            if (exception instanceof OperationInAreaNotPermittedException) {
                // todo: похоже нам нужно уметь различать нехватку прав в группе от остальных для обратной совместимости
                return Option.of(new LegacyMkdirPermissionDeniedException(exception));
            }
        }
        if (origin == LegacyMpfsExceptionOrigin.MOVE) {
            if (exception instanceof InvalidDjfsResourcePathException) {
                return Option.of(new LegacyPathErrorException(exception));
            }
            if (exception instanceof SameSourceAndDestinationException) {
                return Option.of(new LegacyMoveSameDestinationException(exception));
            }
            if (exception instanceof SourceIsParentForDestinationException) {
                return Option.of(new LegacyMoveParentToChildException(exception));
            }
            if (exception instanceof NoParentFolderException) {
                return Option.of(new LegacyMoveNoTargetParentException(exception));
            }
            if (exception instanceof ResourceExistsException) {
                return Option.of(new LegacyResourceAlreadyExists(exception));
            }
            if (exception instanceof PostProcessTaskLimitExceededException) {
                return Option.of(new LegacyTooManyExecutingOperationException(exception));
            }
        }
        return translate(exception, uid);
    }

    @Override
    public int getOrder() {
        return ExceptionHandlerOrders.CUSTOM_EXCEPTION_HANDLER_ORDER;
    }
}
