package ru.yandex.chemodan.app.djfs.core.lenta;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import lombok.Getter;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.InvalidDjfsUidException;
import ru.yandex.misc.lang.Assume;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author yappo
 */

@Getter
public final class AuthorizationClientToken {
    private final String token;
    private final Option<DjfsUid> uid;

    private AuthorizationClientToken(String token, Option<DjfsUid> uid) {
        this.token = token;
        this.uid = uid;
    }

    public static AuthorizationClientToken cons(String rawAuthorizationHeaderValue) {
        Assume assume = new Assume() {
            @Override
            public void fail(String message) {
                throw new InvalidAuthorizationClientTokenHeader(rawAuthorizationHeaderValue);
            }
        };

        Pattern tokenMatchPattern =
                Pattern.compile("^ClientToken\\s+((;?\\s*token=(?<token>[^;$]+))|(;?\\s*uid=(?<uid>[^;$]+)))+;?");
        Matcher matcher = tokenMatchPattern.matcher(rawAuthorizationHeaderValue);
        assume.isTrue(matcher.matches());
        assume.notNull(matcher.group("token"));
        try {
            return new AuthorizationClientToken(matcher.group("token"),
                    Option.ofNullable(matcher.group("uid")).map(DjfsUid::cons));
        } catch (InvalidDjfsUidException e) {
            throw new InvalidAuthorizationClientToken(e);
        }
    }

    public static AuthorizationClientToken cons(HttpServletRequestX req) {
        Assume assume = new Assume() {
            @Override
            public void fail(String message) {
                throw new MissingAuthorizationHeader();
            }
        };

        Option<String> authorizationHeaderO = req.getHeaderO("Authorization");
        assume.isTrue(authorizationHeaderO.isPresent());
        return AuthorizationClientToken.cons(authorizationHeaderO.get());
    }
}
