package ru.yandex.chemodan.app.djfs.core.lenta;

import lombok.RequiredArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.ActionContext;
import ru.yandex.chemodan.app.djfs.core.client.DataapiMordaBlock;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserLocale;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author freyr
 */
@ActionContainer
@RequiredArgsConstructor
public class CoolLentaActions {
    private static final Logger logger = LoggerFactory.getLogger(CoolLentaActions.class);

    private final CoolLentaManager coolLentaManager;

    @Path(value = "/lenta/block_previews", methods = HttpMethod.GET)
    public LentaResultListPojo getBlockPreviews(
            @RequestParam("uid") Option<String> rawUidO,
            @RequestParam("locale") String locale,
            @RequestParam("imageSize") String size,
            @SpecialParam HttpServletRequestX req)
    {
        AuthorizationClientToken clientToken = getAuthorizationClientToken(req);
        coolLentaManager.checkTokenAuthorization(clientToken);
        String clientRawUid = rawUidO.plus(clientToken.getUid().map(DjfsUid::asString))
                .firstO()
                .getOrThrow(LentaBlockPreviewMissingUid::new);
        Option<Dimension> dimension;
        try {
            dimension = Dimension.parse(size);
        } catch (Exception e) {
            logger.info(e);
            return new LentaResultListPojo(Cf.list());
        }
        return coolLentaManager.getBlockPreviews(clientRawUid, locale, dimension);
    }

    @Path(value = "/lenta/block_from_pool", methods = HttpMethod.GET)
    public LentaBlockWithTitlesList getBlocksFromPool(
            @RequestParam("uid") String rawUid,
            @RequestParam("locale") String locale,
            @RequestParam("limit") Option<Integer> limitO,
            @RequestParam("offset") Option<Integer> offsetO)
    {
        DjfsUid uid = DjfsUid.cons(rawUid, ActionContext.CLIENT_INPUT);
        UserLocale userLocale = coolLentaManager.getLocaleWithFallback(locale);

        int offset = offsetO.getOrElse(0);
        int limit = limitO.getOrElse(100);

        ListF<DataapiMordaBlock> blocks = coolLentaManager.fetchBlocksFromPool(uid, offset, limit);
        return new LentaBlockWithTitlesList(limit, offset, blocks.map(
                block -> coolLentaManager.addTitles(userLocale, block)));
    }

    @Path(value = "/lenta/block_info", methods = HttpMethod.GET)
    public LentaBlockInfoWithTitles oneBlockInfo(
            @RequestParam("uid") String rawUid,
            @RequestParam("locale") String locale,
            @RequestParam("blockId") String blockId,
            @RequestParam("fromPool") Option<Boolean> fetchFromPool)
    {
        DjfsUid uid = DjfsUid.cons(rawUid, ActionContext.CLIENT_INPUT);
        UserLocale userLocale = coolLentaManager.getLocaleWithFallback(locale);

        DataapiMordaBlock block = fetchFromPool.getOrElse(false)
                ? coolLentaManager.fetchBlockByIdFromPool(uid, blockId)
                : coolLentaManager.fetchBlockById(uid, blockId);

        return coolLentaManager.addTitles(userLocale, block);
    }

    private AuthorizationClientToken getAuthorizationClientToken(HttpServletRequestX req) {
        try {
            return AuthorizationClientToken.cons(req);
        } catch (InvalidAuthorizationClientToken e) {
            throw new LentaBlockPreviewPermissionDenied(e);
        }
    }
}
