package ru.yandex.chemodan.app.djfs.core.lenta;

import java.net.URI;
import java.util.Comparator;

import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.djfs.core.ActionContext;
import ru.yandex.chemodan.app.djfs.core.DownloadUrlGenerator;
import ru.yandex.chemodan.app.djfs.core.FilenamePreviewStidMimetypeVersionFileId;
import ru.yandex.chemodan.app.djfs.core.client.DataApiHttpClient;
import ru.yandex.chemodan.app.djfs.core.client.DataapiMordaBlock;
import ru.yandex.chemodan.app.djfs.core.client.DataapiMordaBlockList;
import ru.yandex.chemodan.app.djfs.core.client.LentaProperties;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserLocale;
import ru.yandex.chemodan.app.lentaloader.cool.utils.BlockTitlesGenerator;
import ru.yandex.chemodan.app.lentaloader.cool.utils.IntervalType;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TimeIntervalUtils;
import ru.yandex.chemodan.app.lentaloader.cool.utils.TitleGenerationContext;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;

/**
 * @author freyr
 * Taken from CoolLentaActions
 */
@RequiredArgsConstructor
public class CoolLentaManager {
    private static final Logger logger = LoggerFactory.getLogger(CoolLentaManager.class);
    private static final MapF<Language, Function<DataapiMordaBlock, Option<MordaBlockTexts>>> BLOCK_TEXTS_EXTRACTORS = Cf.map(
            Language.ENGLISH, CoolLentaManager::getEnglishTexts,
            Language.RUSSIAN, CoolLentaManager::getRussianTexts,
            Language.TURKISH, CoolLentaManager::getTurkishTexts,
            Language.UKRAINIAN, CoolLentaManager::getUkranianTexts
    );

    private final DjfsResourceDao resourceDao;
    private final DataApiHttpClient dataApiHttpClient;
    private final DownloadUrlGenerator downloadUrlGenerator;
    private final LentaProperties lentaProperties;
    private final ListF<String> expectedInternalTokens;

    private final BlockTitlesGenerator titlesGenerator;

    private final DynamicProperty<Integer> mordaBlocksMaxCountToShow = new DynamicProperty<>("disk-djfs-cool-lenta-max-count-show", 5);
    private final DynamicProperty<Integer> mordaBlocksMaxCountToFetch = new DynamicProperty<>("disk-djfs-cool-lenta-max-count-fetch", 10);

    public LentaResultListPojo getBlockPreviews(String clientRawUid, String locale, Option<Dimension> dimensions) {
        try {
            DjfsUid uid = DjfsUid.cons(clientRawUid, ActionContext.CLIENT_INPUT);
            UserLocale userLocale = UserLocale.R_BY_ALTERNATIVES.valueOf(locale);
            Language language = userLocale.toLanguage();

            if (!titlesGenerator.isSupportedLanguage(language)) {
                return new LentaResultListPojo(Cf.list());
            }

            DataapiMordaBlockList blocks = dataApiHttpClient.fetchNostalgyBlocks(uid, mordaBlocksMaxCountToFetch.get());

            if (blocks.getItems().isEmpty()) {
                return new LentaResultListPojo(Cf.list());
            }

            blocks = filterBlocksByBestResourceId(blocks);

            ListF<DjfsResourceId> allResourceIds = blocks.getItems().
                    flatMap(DataapiMordaBlock::getResourceIds).map(DjfsResourceId::cons);
            if (allResourceIds.map(DjfsResourceId::getUid).exists(x -> !x.equals(uid))) {
                throw new UnexpectedUidInBlockPreviews();
            }

            MapF<String, ListF<String>> bestResourceIdToResourceIds = blocks.getItems().toMap(
                    DataapiMordaBlock::getBestResourceId,
                    block -> Cf.list(block.getBestResourceId())
                            .plus(block.getResourceIds().findNot(block.getBestResourceId()::equals)));

            ListF<DjfsResourceId> filteredResourceIds =
                    bestResourceIdToResourceIds.values().flatMap(x -> x).map(DjfsResourceId::cons);

            ListF<FilenamePreviewStidMimetypeVersionFileId> meta =
                    resourceDao.find2FilenamePreviewStidMimetypeVersionFid(
                            uid, filteredResourceIds.map(DjfsResourceId::getFileId)
                    );

            Comparator<FilenamePreviewStidMimetypeVersionFileId> comparator = Comparator
                    .<FilenamePreviewStidMimetypeVersionFileId>comparingLong(x -> x.getVersion().getOrElse(0L))
                    .reversed()
                    .thenComparing(FilenamePreviewStidMimetypeVersionFileId::getId);

            ListF<FilenamePreviewStidMimetypeVersionFileId> uniqueResourcesMeta =
                    meta.groupBy(FilenamePreviewStidMimetypeVersionFileId::getFileId)
                            .mapValues(x -> x.sorted(comparator).first()).values().toList();

            MapF<String, FilenamePreviewStidMimetypeVersionFileId> resourceIdToMeta = uniqueResourcesMeta.toMap(
                    x -> DjfsResourceId.cons(uid, x.getFileId().get()).getValue(),
                    x -> x
            );

            ListF<LentaResultItemPojo> items = Cf.arrayList();
            for (DataapiMordaBlock block : blocks.getItems()) {
                if (bestResourceIdToResourceIds.getO(block.getBestResourceId()).get()
                        .exists(x -> !resourceIdToMeta.getO(x).isPresent()))
                {
                    continue;  // skip the whole block
                }

                Option<FilenamePreviewStidMimetypeVersionFileId> resourceMetaO = resourceIdToMeta.getO(
                        block.getBestResourceId()
                );
                if (!resourceMetaO.isPresent() || !resourceMetaO.get().getPreviewStid().isPresent()) {
                    continue;  // skip the whole block
                }
                FilenamePreviewStidMimetypeVersionFileId resourceMeta = resourceMetaO.get();

                // should never be used for shared folders, so we consider owner_uid to be same as uid
                URI previewUri = downloadUrlGenerator.makePublicPreviewUrl(() -> uid,
                        resourceMeta.getPreviewStid().get(), resourceMeta.getMimetype(), resourceMeta.getFilename(),
                        dimensions, true, false
                );

                MordaBlockTexts texts = BLOCK_TEXTS_EXTRACTORS.getTs(language).apply(block)
                        .getOrElse(() -> generateDefaultTextsForBlock(buildTitleGenerationContext(block), language));

                items.add(new LentaResultItemPojo(block.getId(), previewUri, texts.getTitle(), Option.of(texts.getSubtitle())));
            }

            if (items.length() < lentaProperties.getMinBlockCount()) {
                return new LentaResultListPojo(Cf.list());
            }
            return new LentaResultListPojo(items);
        } catch (Exception e) {
            logger.info(e);
            return new LentaResultListPojo(Cf.list());
        }
    }

    public ListF<DataapiMordaBlock> fetchBlocksFromPool(DjfsUid uid, int offset, int limit) {
        return dataApiHttpClient.fetchBlocksFromPool(uid, offset, limit).getItems();
    }

    public DataapiMordaBlock fetchBlockByIdFromPool(DjfsUid uid, String blockId) {
        return dataApiHttpClient.fetchBlockByIdFromPool(uid, blockId);
    }

    public DataapiMordaBlock fetchBlockById(DjfsUid uid, String blockId) {
        return dataApiHttpClient.fetchBlockById(uid, blockId);
    }

    private static Option<MordaBlockTexts> getRussianTexts(DataapiMordaBlock mordaBlock) {
        if (Cf.list(mordaBlock.getTitleRu(), mordaBlock.getSubtitleRu(), mordaBlock.getPhotosliceLinkTextRu())
                .exists(textO -> !textO.isPresent())) {
            return Option.empty();
        }
        return Option.of(new MordaBlockTexts(mordaBlock.getTitleRu().get(), mordaBlock.getSubtitleRu().get(), mordaBlock.getPhotosliceLinkTextRu().get()));
    }

    private static Option<MordaBlockTexts> getEnglishTexts(DataapiMordaBlock mordaBlock) {
        if (Cf.list(mordaBlock.getTitleEn(), mordaBlock.getSubtitleEn(), mordaBlock.getPhotosliceLinkTextEn())
                .exists(textO -> !textO.isPresent())) {
            return Option.empty();
        }
        return Option.of(new MordaBlockTexts(mordaBlock.getTitleEn().get(), mordaBlock.getSubtitleEn().get(),
                mordaBlock.getPhotosliceLinkTextEn().get()));
    }

    private static Option<MordaBlockTexts> getTurkishTexts(DataapiMordaBlock mordaBlock) {
        if (Cf.list(mordaBlock.getTitleTr(), mordaBlock.getSubtitleTr(), mordaBlock.getPhotosliceLinkTextTr())
                .exists(textO -> !textO.isPresent())) {
            return Option.empty();
        }
        return Option.of(new MordaBlockTexts(mordaBlock.getTitleTr().get(), mordaBlock.getSubtitleTr().get(),
                mordaBlock.getPhotosliceLinkTextTr().get()));
    }

    private static Option<MordaBlockTexts> getUkranianTexts(DataapiMordaBlock mordaBlock) {
        if (Cf.list(mordaBlock.getTitleUk(), mordaBlock.getSubtitleUk(), mordaBlock.getPhotosliceLinkTextUk())
                .exists(textO -> !textO.isPresent())) {
            return Option.empty();
        }
        return Option.of(new MordaBlockTexts(mordaBlock.getTitleUk().get(), mordaBlock.getSubtitleUk().get(),
                mordaBlock.getPhotosliceLinkTextUk().get()));
    }

    @NotNull
    public LentaBlockInfoWithTitles addTitles(UserLocale userLocale, DataapiMordaBlock block) {
        Language language = userLocale.toLanguage();
        MordaBlockTexts blockTexts = BLOCK_TEXTS_EXTRACTORS.getTs(language).apply(block)
                .getOrElse(() -> generateDefaultTextsForBlock(buildTitleGenerationContext(block), language));

        return new LentaBlockInfoWithTitles(block, blockTexts.getTitle(), Option.of(blockTexts.getSubtitle()),
                Option.of(blockTexts.getPhotosliceLinkText()));
    }

    private TitleGenerationContext buildTitleGenerationContext(DataapiMordaBlock block) {
        IntervalType intervalType = TimeIntervalUtils.getMinimalInterval(block.getUserMinDate(), block.getUserMaxDate());
        DateTime intervalStart = intervalType.getIntervalStart(block.getUserMinDate());
        return new TitleGenerationContext(getRandomForBlock(block), intervalType, intervalStart);
    }

    private MordaBlockTexts generateDefaultTextsForBlock(TitleGenerationContext context, Language language) {
        return new MordaBlockTexts(titlesGenerator.generateBlockTitle(context).get(language),
                titlesGenerator.generateBlockSubtitle(context).get(language),
                titlesGenerator.generatePhotosliceLinkText(context).get(language));
    }

    public UserLocale getLocaleWithFallback(@RequestParam("locale") String locale) {
        UserLocale userLocale = UserLocale.R_BY_ALTERNATIVES.valueOf(locale);
        if (!titlesGenerator.isSupportedLanguage(userLocale.toLanguage())) {
            return UserLocale.EN;
        }
        return userLocale;
    }

    @NotNull
    private Random2 getRandomForBlock(DataapiMordaBlock block) {
        return new Random2(block.getBestResourceId().hashCode());
    }

    private DataapiMordaBlockList filterBlocksByBestResourceId(DataapiMordaBlockList blocks) {
        ListF<DataapiMordaBlock> filteredBlocks = Cf.arrayList();
        for (DataapiMordaBlock block : blocks.getItems()) {
            if (filteredBlocks.map(BaseMordaBlockData::getBestResourceId).containsTs(block.getBestResourceId())) {
                continue;
            }
            filteredBlocks.add(block);
            if (filteredBlocks.size() == mordaBlocksMaxCountToShow.get()) {
                break;
            }
        }
        return new DataapiMordaBlockList(Cf.x(filteredBlocks));
    }


    public void checkTokenAuthorization(AuthorizationClientToken tokenFromClient) {
        if (!expectedInternalTokens.exists(t -> t.equals(tokenFromClient.getToken()))) {
            throw new LentaBlockPreviewPermissionDenied();
        }
    }
}
