package ru.yandex.chemodan.app.djfs.core.lenta.alice;

import com.google.protobuf.Struct;
import lombok.AllArgsConstructor;
import lombok.Data;

import ru.yandex.alice.megamind.protos.scenarios.RequestProto;
import ru.yandex.alice.megamind.protos.scenarios.ResponseProto;
import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.alice.AliceScenario;
import ru.yandex.chemodan.alice.AliceUtils;
import ru.yandex.chemodan.alice.ClientInfo;
import ru.yandex.chemodan.alice.DivTemplateProcessor;
import ru.yandex.chemodan.app.djfs.core.lenta.CoolLentaManager;
import ru.yandex.chemodan.app.djfs.core.lenta.LentaResultItemPojo;
import ru.yandex.chemodan.app.djfs.core.lenta.LentaResultListPojo;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;

/**
 * @author yashunsky
 */
@AllArgsConstructor
public class MyPhotosScenario implements AliceScenario {
    private static final Logger logger = LoggerFactory.getLogger(MyPhotosScenario.class);

    private final DynamicProperty<String> authFailureText = new DynamicProperty<>("alice.my-photos.auth-failure",
            "Не могу открыть ваши фото. " +
                    "Чтобы всё получилось, авторизуйтесь в Яндексе и загрузите фотографии на Яндекс Диск");

    private final DynamicProperty<String> noPhotosText = new DynamicProperty<>("alice.my-photos.no-photos",
            "Не могу открыть ваши фото. " +
                    "Чтобы всё получилось, авторизуйтесь в Яндексе и загрузите фотографии на Яндекс Диск");

    private final DynamicProperty<ListF<String>> successText = new DynamicProperty<>("alice.my-photos.success",
            Cf.list("Составила подборки из ваших снимков. А вы неплохой фотограф!",
                    "Вы сделали красивые фото — я составила из них подборки. Мы оба молодцы!",
                    "Ваши фото хороши, а в подборках они еще лучше. Посмотрите сами!"));

    private final DynamicProperty<String> imageSize = new DynamicProperty<>("alice.image-size", "544x408");

    private final CoolLentaManager coolLentaManager;

    private final DivTemplateProcessor templateProcessor =
            new DivTemplateProcessor(getClass(), "list_photo_selections.ftj");
    @Override
    public String getIntentName() {
        return "personal_assistant.scenarios.disk_my_photos";
    }

    @Override
    public ResponseProto.TScenarioRunResponse run(Option<Long> uidO, RequestProto.TScenarioRunRequest request) {
        ClientInfo info = ClientInfo.fromProto(request.getBaseRequest().getClientInfo());
        if (!(info.isSearchApp() || info.isYaBrowser())) {
            return AliceUtils.irrelevantResponse();
        }

        if (!uidO.isPresent()) {
            logger.warn("No uid provided");
            return AliceUtils.createSimpleTextResponse(authFailureText.get());
        }

        LentaResultListPojo lentaResults =
                coolLentaManager.getBlockPreviews(String.valueOf(uidO.get()), "ru", Dimension.parse(imageSize.get()));

        ListF<BlockModel> blockModels = lentaResults.items.getOrElse(Cf.list())
                .zipWithIndex().map(BlockModel::fromLentaResultItem);

        if (blockModels.isEmpty()) {
            logger.warn("No items found for uid {}", uidO.get());
            return AliceUtils.createSimpleTextResponse(noPhotosText.get());
        }

        Struct divJson = templateProcessor.processTemplate(Cf.map("blocks", blockModels));
        logger.info("div json: {}", divJson);

        return ResponseProto.TScenarioRunResponse
                .newBuilder()
                .setResponseBody(ResponseProto.TScenarioResponseBody
                        .newBuilder()
                        .setLayout(ResponseProto.TLayout
                                        .newBuilder()
                                        .addCards(ResponseProto.TLayout.TCard.newBuilder().setDivCard(divJson))
                                .setOutputSpeech(Random2.R.randomElement(successText.get()))
                        ))
                .build();
    }

    @Data
    public static class BlockModel {
        public final String imageUrl;
        public final String title;
        public final String subtitle;
        public final String link;
        public final int index;

        public static BlockModel fromLentaResultItem(LentaResultItemPojo item, int index) {
            return new BlockModel(item.imageUrl, item.title1, item.title2.getOrElse(""), item.link, index);
        }
    }
}
