package ru.yandex.chemodan.app.djfs.core.lock;

import java.util.Objects;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.db.DjfsShardInfo;
import ru.yandex.chemodan.app.djfs.core.db.EntityAlreadyExistsException;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;

/**
 * Handles only temporary filesystem restrictions.
 * Checks path locks, migration locks, read-only status.
 * <p>
 * Permanent filesystem restrictions are handled by FilesystemAccessController.
 *
 * @author eoshch
 * @see ru.yandex.chemodan.app.djfs.core.filesystem.FilesystemAccessController
 */
public class LockManager {
    private final FilesystemLockDao filesystemLockDao;
    private final PgMigrationLockDao pgMigrationLockDao;

    public LockManager(FilesystemLockDao filesystemLockDao, PgMigrationLockDao pgMigrationLockDao)
    {
        this.filesystemLockDao = filesystemLockDao;
        this.pgMigrationLockDao = pgMigrationLockDao;
    }

    public boolean isLocked(DjfsUid uid) {
        return pgMigrationLockDao.isLocked(uid);
    }

    public boolean isLocked(DjfsResourcePath path) {
        return !getLocks(path).isEmpty();
    }

    public boolean isLockedFor(String lockerId, DjfsResourcePath path) {
        return !getLocks(path).filter(x -> !Objects.equals(x.getLockerId().getOrElse(""), lockerId)).isEmpty();
    }

    public ListF<FilesystemLock> getLocks(DjfsResourcePath path) {
        ListF<FilesystemLock> locks = filesystemLockDao.find(path.getUid());
        return locks.filter(x ->
                Objects.equals(x.getPath().getPath(), path.getPath())
                        || x.getPath().isParentFor(path)
                        || path.isParentFor(x.getPath()));
    }

    public void checkUserAndPath(DjfsResourcePath path) {
        if (isLocked(path.getUid())) {
            throw new UserLockedException(path.getUid());
        }
        ListF<FilesystemLock> lock = getLocks(path);
        if (!lock.isEmpty()) {
            throw new ResourceLockedException(lock.first().getPath());
        }
    }

    public boolean tryAcquireOrRenewLock(String lockerId, String operationId, String operationType,
            DjfsResourcePath path)
    {
        //todo: make one request to DB instead of two
        try {
            FilesystemLock lock = FilesystemLock.builder()
                    .path(path)
                    .lockerId(Option.of(lockerId))
                    .operationId(Option.of(operationId))
                    .operationType(Option.of(operationType))
                    .dtime(Instant.now())
                    .build();
            filesystemLockDao.insert(lock);
            return true;
        } catch (EntityAlreadyExistsException e) {
            return filesystemLockDao.update(path, lockerId, operationId, operationType, Instant.now());
        }
    }

    public void unlock(String lockerId, DjfsResourcePath path) {
        filesystemLockDao.delete(path, lockerId);
    }

    public void lock(DjfsResourcePath path) {
        try {
            filesystemLockDao.insert(FilesystemLock.builder().path(path).dtime(Instant.now()).build());
        } catch (EntityAlreadyExistsException e) {
            throw new ResourceLockedException(path, e);
        }
    }

    public void updateLock(DjfsResourcePath path) {
        filesystemLockDao.setDtime(path, Instant.now());
    }

    public void unlock(DjfsResourcePath path) {
        filesystemLockDao.delete(path);
    }

    public void lockForMigration(DjfsUid uid, DjfsShardInfo.Pg shardInfo, Duration lockFor, String ownerMark) {
        pgMigrationLockDao.insertOrUpdate(uid, shardInfo, Instant.now().plus(lockFor), ownerMark);
    }

    public void unlockForMigration(DjfsUid uid, DjfsShardInfo.Pg shardInfo, String ownerMark) {
        pgMigrationLockDao.delete(uid, shardInfo, ownerMark);
    }

    public void updateLockForMigration(DjfsUid uid, DjfsShardInfo.Pg shardInfo, Duration lockFor, String ownerMark) {
        pgMigrationLockDao.setUntil(uid, shardInfo, Instant.now().plus(lockFor), ownerMark);
    }

    public boolean isLockedForMigration(DjfsUid uid, DjfsShardInfo.Pg shardInfo) {
        return pgMigrationLockDao.isLocked(uid, shardInfo);
    }

    public void cleanUpMigrationLock(DjfsUid uid, DjfsShardInfo.Pg shardInfo) {
        pgMigrationLockDao.deleteExpired(uid, shardInfo);
    }

    public void cleanUpMigrationLocks(DjfsShardInfo.Pg shardInfo, Instant olderThan) {
        pgMigrationLockDao.deleteAllExpired(shardInfo, olderThan);
    }
}
