package ru.yandex.chemodan.app.djfs.core.lock;

import com.mongodb.ErrorCategory;
import com.mongodb.MongoWriteException;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.Updates;
import com.mongodb.client.result.UpdateResult;
import org.bson.BsonDocument;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.db.EntityAlreadyExistsException;
import ru.yandex.chemodan.app.djfs.core.db.mongo.MongoShardedDao;
import ru.yandex.chemodan.app.djfs.core.db.mongo.MongoShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.db.mongo.MongoUtil;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author eoshch
 */
public class MongoFilesystemLockDao extends MongoShardedDao<String, MongoFilesystemLock> implements FilesystemLockDao {
    private static final Logger logger = LoggerFactory.getLogger(MongoFilesystemLockDao.class);

    public MongoFilesystemLockDao(MongoShardedDaoContext context) {
        super(context, "filesystem_locks", "filesystem_locks", MongoFilesystemLock.B);
    }

    @Override
    public void insert(FilesystemLock lock) {
        // todo: check dtime timezone
        MongoFilesystemLock entity = MongoFilesystemLock.from(lock);
        try {
            collectionX(lock.getUid()).insertOne(entity);
        } catch (MongoWriteException e) {
            if (ErrorCategory.fromErrorCode((e.getError().getCode())) == ErrorCategory.DUPLICATE_KEY) {
                logger.warn("MongoFilesystemLockDao.insert(FilesystemLock) handled exception for path "
                        + lock.getPath().toString() + " : ", e);
                throw new EntityAlreadyExistsException(entity.id, e);
            }
            throw e;
        }
    }

    public boolean update(DjfsResourcePath path, String lockerId, String operationId, String operationType,
            Instant dtime)
    {
        MongoCollection<BsonDocument> collection = collection(path);

        UpdateResult r = MongoUtil.retryTimeoutOnce(
                () -> collection.updateOne(
                        Filters.and(
                                Filters.eq("_id", path.getMongoId()),
                                Filters.eq("uid", path.getUid().asString()),
                                Filters.eq("key", path.getPath()),
                                Filters.eq("data.locker_id", lockerId)
                        ), Updates.combine(
                                Updates.set("dtime", dtime.toDate()),
                                Updates.set("data.oid", operationId),
                                Updates.set("data.op_type", operationType))));
        return r.getModifiedCount() > 0;
    }

    @Override
    public void setDtime(DjfsResourcePath path, Instant dtime) {
        collectionX(path).updateOne(
                Filters.and(
                        Filters.eq("_id", path.getMongoId()),
                        Filters.eq("uid", path.getUid().asString()),
                        Filters.eq("key", path.getPath())),
                Updates.set("dtime", dtime.toDate()));
    }

    @Override
    public ListF<FilesystemLock> find(DjfsUid uid) {
        return collectionX(uid).find(Filters.eq("uid", uid.asString())).map(MongoFilesystemLock::to);
    }

    @Override
    public Option<FilesystemLock> find(DjfsResourcePath path) {
        return collectionX(path).findOne(Filters.and(
                Filters.eq("_id", path.getMongoId()),
                Filters.eq("uid", path.getUid().asString()),
                Filters.eq("key", path.getPath())
        )).map(MongoFilesystemLock::to);
    }

    @Override
    public void delete(DjfsResourcePath path) {
        collectionX(path.getUid()).deleteOne(path.getMongoId());
    }

    @Override
    public void delete(DjfsResourcePath path, String lockerId) {
        collectionX(path.getUid()).deleteOne(Filters.and(
                Filters.eq("_id", path.getMongoId()),
                Filters.eq("uid", path.getUid().asString()),
                Filters.eq("key", path.getPath()),
                Filters.eq("data.locker_id", lockerId)
        ));
    }
}
