package ru.yandex.chemodan.app.djfs.core.lock;

import java.util.Map;
import java.util.Objects;

import com.google.gson.Gson;
import org.joda.time.Instant;
import org.postgresql.util.PSQLException;
import org.postgresql.util.ServerErrorMessage;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.djfs.core.db.EntityAlreadyExistsException;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDao;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author eoshch
 */
public class PgFilesystemLockDao extends PgShardedDao implements FilesystemLockDao {
    private static final Logger logger = LoggerFactory.getLogger(PgFilesystemLockDao.class);
    private static final Gson gson = new Gson();

    private final static RowMapper<FilesystemLock> M = (rs, rowNum) -> {
        Option<String> lockerId;
        Option<String> operationId;
        Option<String> operationType;

        String data = rs.getString("data");
        if (StringUtils.isNotBlank(data)) {
            Map map = gson.fromJson(data, Map.class);
            lockerId = Option.ofNullable((String) map.get("locker_id"));
            operationId = Option.ofNullable((String) map.get("oid"));
            operationType = Option.ofNullable((String) map.get("op_type"));
        } else {
            lockerId = Option.empty();
            operationId = Option.empty();
            operationType = Option.empty();
        }

        return FilesystemLock.builder()
                .path(DjfsResourcePath.cons(rs.getLong("uid"), rs.getString("path")))
                .lockerId(lockerId)
                .operationId(operationId)
                .operationType(operationType)
                .dtime(new Instant(rs.getTimestamp("dtime")))
                .build();
    };


    public PgFilesystemLockDao(PgShardedDaoContext context) {
        super(context);
    }

    @Override
    public void insert(FilesystemLock lock) {
        // todo: check dtime timezone
        String sql = collectStats(lock) + " INSERT INTO disk.filesystem_locks (id, uid, path, dtime, data) "
                + " VALUES (:id, :uid, :path, :dtime, :data::json)";

        Map<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "id", lock.getPath().getPgId(),
                "uid", lock.getPath().getUid(),
                "path", lock.getPath().getPath(),
                "dtime", lock.getDtime()
        ));

        if (lock.getLockerId().isPresent() || lock.getOperationId().isPresent() || lock.getOperationType().isPresent()) {
            MapF<String, String> map = Cf.hashMap();
            if (lock.getLockerId().isPresent()) {
                map.put("locker_id", lock.getLockerId().get());
            }
            if (lock.getOperationId().isPresent()) {
                map.put("oid", lock.getOperationId().get());
            }
            if (lock.getOperationType().isPresent()) {
                map.put("op_type", lock.getOperationType().get());
            }
            parameters.put("data", gson.toJson(map));
        } else {
            parameters.put("data", null);
        }

        try {
            jdbcTemplate(lock.getUid()).update(sql, parameters);
        } catch (DataIntegrityViolationException e) {
            Throwable cause = e.getCause();
            if (cause instanceof PSQLException) {
                ServerErrorMessage error = ((PSQLException) cause).getServerErrorMessage();
                if (Objects.equals(error.getConstraint(), "pk_filesystem_locks")) {
                    logger.warn("PgFilesystemLockDao.insert(FilesystemLock) handled exception for path "
                            + lock.getPath().toString() + " : ", e);
                    throw new EntityAlreadyExistsException(lock.getPath().getMongoId(), e);
                }
            }
            throw e;
        }
    }

    @Override
    public void setDtime(DjfsResourcePath path, Instant dtime) {
        String sql = collectStats(path)
                + " UPDATE disk.filesystem_locks SET dtime = :dtime WHERE id = :id AND uid = :uid AND path = :path";
        Map<String, Object> parameters = Cf.map(
                "id", path.getPgId(),
                "uid", path.getUid(),
                "path", path.getPath(),
                "dtime", dtime);
        jdbcTemplate(path).update(sql, parameters);
    }

    @Override
    public boolean update(DjfsResourcePath path, String lockerId, String operationId, String operationType, Instant dtime) {
        String sql = collectStats(path) + " UPDATE disk.filesystem_locks SET dtime = :dtime, "
                + "     data = jsonb_set(jsonb_set(data, '{oid}', :operation_id::jsonb), '{op_type}', :operation_type::jsonb)"
                + " WHERE id = :id AND uid = :uid AND path = :path AND data->>'locker_id' = :locker_id";

        String sql2 = collectStats(path) + " SELECT id, uid, path, dtime, data FROM disk.filesystem_locks WHERE uid = ?";
        jdbcTemplate(path).query(sql2, M, path.getUid());

        Map<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "id", path.getPgId(),
                "uid", path.getUid(),
                "path", path.getPath(),
                "dtime", dtime,
                "locker_id", lockerId,
                "operation_id", gson.toJson(operationId),
                "operation_type", gson.toJson(operationType)));
        int updated = jdbcTemplate(path).update(sql, parameters);
        return updated > 0;
    }

    @Override
    public ListF<FilesystemLock> find(DjfsUid uid) {
        String sql = collectStats(uid) + " SELECT id, uid, path, dtime, data FROM disk.filesystem_locks WHERE uid = ?";
        return jdbcTemplate(uid).query(sql, M, uid);
    }

    @Override
    public Option<FilesystemLock> find(DjfsResourcePath path) {
        String sql = collectStats(path) + " SELECT id, uid, path, dtime, data FROM disk.filesystem_locks "
                + " WHERE id = :id AND uid = :uid AND path = :path";
        Map<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "id", path.getPgId(),
                "uid", path.getUid(),
                "path", path.getPath()));
        return jdbcTemplate(path).queryForOption(sql, M, parameters);
    }

    @Override
    public void delete(DjfsResourcePath path) {
        String sql = collectStats(path) + " DELETE FROM disk.filesystem_locks WHERE id = ?";
        jdbcTemplate(path.getUid()).update(sql, path.getPgId());
    }

    @Override
    public void delete(DjfsResourcePath path, String lockerId) {
        String sql = collectStats(path) + " DELETE FROM disk.filesystem_locks "
                + " WHERE id = :id AND uid = :uid AND path = :path AND data->>'locker_id' = :locker_id";
        Map<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "id", path.getPgId(),
                "uid", path.getUid(),
                "path", path.getPath(),
                "locker_id", lockerId));
        jdbcTemplate(path.getUid()).update(sql, parameters);
    }
}
