package ru.yandex.chemodan.app.djfs.core.lock;

import org.joda.time.Instant;
import org.springframework.dao.support.DataAccessUtils;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.chemodan.app.djfs.core.db.DjfsShardInfo;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDao;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class PgMigrationLockDao extends PgShardedDao {
    public PgMigrationLockDao(PgShardedDaoContext context) {
        super(context);
    }

    /**
     * Записываем лок. Если лок уже существем, проверям, что старый лок протух. В противном случае валимся.
     */
    public void insertOrUpdate(DjfsUid uid, DjfsShardInfo.Pg shardInfo, Instant until, String ownerMark) {
        boolean inserted = DataAccessUtils.requiredSingleResult(jdbcTemplate(shardInfo).query(""
                        + "WITH ins AS ("
                        + "     INSERT INTO disk.migration_lock(uid, lock_until, owner_mark) "
                        + "             VALUES (:uid, :until, :ownerMark)"
                        + "         ON CONFLICT (uid)"
                        + "     DO UPDATE SET lock_until = :until, owner_mark = :ownerMark "
                        + "         WHERE migration_lock.lock_until < now()"
                        + "         RETURNING lock_until"
                        + ")"
                        + "SELECT disk.get_migration_lock(:uid), exists(SELECT 1 from ins) as inserted",
                (rs, i) -> rs.getBoolean("inserted"),
                Cf.map(
                        "uid", uid,
                        "until", until,
                        "ownerMark", ownerMark
                )
        ));
        if (!inserted) {
            throw new IllegalStateException("user already locked for migration");
        }
    }

    public void delete(DjfsUid uid, DjfsShardInfo.Pg shardInfo, String ownerMark) {
        boolean deleted = DataAccessUtils.requiredSingleResult(jdbcTemplate(shardInfo).query(""
                        + "WITH del(uid) AS ("
                        + "     DELETE FROM disk.migration_lock "
                        + "         WHERE uid = :uid AND owner_mark = :ownerMark AND lock_until >= now()"
                        + "     RETURNING uid"
                        + ")"
                        + "SELECT disk.get_migration_lock(:uid), exists(SELECT 1 from del) as deleted",
                (rs, i) -> rs.getBoolean("deleted"),
                Cf.map(
                        "uid", uid,
                        "ownerMark", ownerMark
                )
        ));
        if (!deleted) {
            throw new IllegalStateException("can not delete lock. Lock owner is different or lock expired");
        }
    }

    public void deleteExpired(DjfsUid uid, DjfsShardInfo.Pg shardInfo) {
        jdbcTemplate(shardInfo).query(""
                        + "WITH del(uid) AS ("
                        + "     DELETE FROM disk.migration_lock "
                        + "         WHERE uid = :uid AND lock_until < now()"
                        + "     RETURNING uid"
                        + ")"
                        + "SELECT disk.get_migration_lock(:uid), exists(SELECT 1 from del) as deleted",
                (rs, i) -> rs.getBoolean("deleted"),
                Cf.map(
                        "uid", uid
                )
        );
    }

    public void deleteAllExpired(DjfsShardInfo.Pg shardInfo, Instant olderThan) {
        jdbcTemplate(shardInfo).update(
                "DELETE FROM disk.migration_lock WHERE lock_until <= :olderThan",
                Cf.map("olderThan", olderThan)
        );
    }

    public void setUntil(DjfsUid uid, DjfsShardInfo.Pg shardInfo, Instant until, String ownerMark) {
        boolean updated = DataAccessUtils.requiredSingleResult(jdbcTemplate(shardInfo).query(""
                        + "WITH upd(uid) AS ("
                        + "     UPDATE disk.migration_lock SET lock_until = :until "
                        + "         WHERE uid = :uid AND owner_mark = :ownerMark AND lock_until >= now()"
                        + "     RETURNING uid"
                        + ")"
                        + "SELECT disk.get_migration_lock(:uid), exists(SELECT 1 from upd) as updated",
                (rs, i) -> rs.getBoolean("updated"),
                Cf.map(
                        "uid", uid,
                        "until", until,
                        "ownerMark", ownerMark
                )
        ));
        if (!updated) {
            throw new IllegalStateException("can not update lock. Lock owner is different or lock expired");
        }
    }

    public boolean isLocked(DjfsUid uid) {
        return isLocked(jdbcTemplate(uid), uid);
    }

    public boolean isLocked(DjfsUid uid, DjfsShardInfo.Pg shardInfo) {
        return isLocked(jdbcTemplate(shardInfo), uid);
    }

    private boolean isLocked(JdbcTemplate3 shard, DjfsUid uid) {
        return shard.queryForObject(
                "SELECT disk.is_locked_for_migration(?, ?)",
                Boolean.class,
                uid,
                Instant.now()
        );
    }
}
