package ru.yandex.chemodan.app.djfs.core.notification;

import java.util.concurrent.ExecutorService;

import lombok.RequiredArgsConstructor;

import ru.yandex.chemodan.app.djfs.core.EventManager;
import ru.yandex.chemodan.app.djfs.core.filesystem.QuotaInfo;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.BlackboxUtils;
import ru.yandex.chemodan.app.djfs.core.share.event.UserKickedFromGroupEvent;
import ru.yandex.chemodan.app.djfs.core.share.event.UserLeftGroupEvent;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.CeleryJobUtils;
import ru.yandex.chemodan.app.djfs.core.util.DjfsAsyncTaskUtils;
import ru.yandex.chemodan.app.djfs.core.util.ExecutorServiceUtils;
import ru.yandex.chemodan.app.djfs.core.util.JsonUtils;
import ru.yandex.chemodan.queller.celery.job.CeleryJob;
import ru.yandex.chemodan.queller.worker.CeleryTaskManager;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxDbFields;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxDisplayName;
import ru.yandex.misc.email.Email;

/**
 * @author eoshch
 */
@RequiredArgsConstructor
public class EmailGenerator {
    public static TaskId TASK_ID = new TaskId("mpfs.core.job_handlers.routine.handle_send_email");
    public static TaskId SPACE_TASK_ID = new TaskId("mpfs.core.job_handlers.routine.handle_send_email_space");

    private final ExecutorService executorService;
    private final CeleryTaskManager celeryTaskManager;
    private final Blackbox2 blackbox2;

    @EventManager.EventHandler
    public void onUserLeftGroup(UserLeftGroupEvent event) {
        BlackboxCorrectResponse owner = BlackboxUtils.getBlackboxUserInfo(event.getShareInfo().getOwnerUid(), blackbox2);

        if (!owner.getUid().isPresent()) {
            return;
        }

        if (!owner.getDefaultEmail().isPresent()) {
            return;
        }

        BlackboxCorrectResponse participant = BlackboxUtils.getBlackboxUserInfo(event.getUid(), blackbox2);

        String participantEmail = participant.getDefaultEmail().map(Email::getEmail).getOrElse("");
        String participantName = participant.getDisplayName().filterMap(BlackboxDisplayName::getPublicName).getOrElse("");
        String ownerLocale = owner.getDbFields().getOrElse(BlackboxDbFields.LANG, "");

        JsonUtils.JsonObjectBuilder kwargsBuilder = JsonUtils.objectBuilder()
                .add("sender_email", participantEmail)
                .add("sender_name", participantName)
                .add("template_name", "sharedFolder/left")
                .add("template_args", JsonUtils.objectBuilder()
                        .add("username", participantName)
                        .add("locale", ownerLocale)
                        .add("userEmail", participantEmail)
                        .add("gid", event.getShareInfo().getGroupId())
                        .add("folderName", event.getShareInfo().getGroupPath().getName())
                        .toJsonObject())
                .add("email_to", owner.getDefaultEmail().get().getEmail());

        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, kwargsBuilder.toMap());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));
    }

    @EventManager.EventHandler
    public void onUserKickedFromGroup(UserKickedFromGroupEvent event) {
        BlackboxCorrectResponse participant = BlackboxUtils.getBlackboxUserInfo(event.getUid(), blackbox2);

        if (!participant.getUid().isPresent()) {
            return;
        }

        if (!participant.getDefaultEmail().isPresent()) {
            return;
        }

        BlackboxCorrectResponse owner = BlackboxUtils.getBlackboxUserInfo(event.getUid(), blackbox2);

        String ownerEmail = owner.getDefaultEmail().map(Email::getEmail).getOrElse("");
        String ownerName = owner.getDisplayName().filterMap(BlackboxDisplayName::getPublicName).getOrElse("");
        String participantEmail = participant.getDefaultEmail().get().getEmail();
        String participantLocale = participant.getDbFields().getOrElse(BlackboxDbFields.LANG, "");
        String participantFolderName = event.getShareInfo().getRootPath(event.getUid())
                .map(DjfsResourcePath::getName).getOrElse("");

        JsonUtils.JsonObjectBuilder kwargsBuilder = JsonUtils.objectBuilder()
                .add("sender_email", ownerEmail)
                .add("sender_name", ownerName)
                .add("template_name", "sharedFolder/kicked")
                .add("template_args", JsonUtils.objectBuilder()
                        .add("locale", participantLocale)
                        .add("gid", event.getShareInfo().getGroupId())
                        .add("friendName", ownerName)
                        .add("folderName", participantFolderName)
                        .add("friendEmail", ownerEmail)
                        .toJsonObject())
                .add("email_to", participantEmail);

        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, kwargsBuilder.toMap());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));
    }

    private JsonObject spaceData(DjfsUid uid, QuotaInfo quotaInfo, long fileCount) {
        return JsonUtils.objectBuilder()
                .add("uid", uid.asString())
                .add("limit", quotaInfo.getLimit())
                .add("used", quotaInfo.getUsed())
                .add("free", quotaInfo.getFree())
                .add("trash", quotaInfo.getUsedByTrash())
                .add("files_count", fileCount)
                .add("filesize_limit", DjfsResourceArea.DISK.singleFileSizeLimit())
                .toJsonObject();
    }

    public void sendLowFreeSpaceMail(DjfsUid uid, QuotaInfo quotaInfo, long fileCount) {
        JsonUtils.JsonObjectBuilder kwargsBuilder = JsonUtils.objectBuilder()
                .add("uid", uid.asString())
                .add("template", "space/fastFinished")
                .add("data", spaceData(uid, quotaInfo, fileCount));

        String deduplicationId = DjfsAsyncTaskUtils.activeUid("send_email_space__space/fastFinished_" + uid.asString());

        CeleryJob celeryJob = CeleryJobUtils.create(SPACE_TASK_ID, deduplicationId, kwargsBuilder.toMap());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));
    }

    public void sendNoFreeSpaceMail(DjfsUid uid, QuotaInfo quotaInfo, long fileCount) {
        JsonUtils.JsonObjectBuilder kwargsBuilder = JsonUtils.objectBuilder()
                .add("uid", uid.asString())
                .add("template", "space/finished")
                .add("data", spaceData(uid, quotaInfo, fileCount));

        String deduplicationId = DjfsAsyncTaskUtils.activeUid("send_email_space__space/finished_" + uid.asString());

        CeleryJob celeryJob = CeleryJobUtils.create(SPACE_TASK_ID, deduplicationId, kwargsBuilder.toMap());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));
    }

    public void sendNoFreeSpaceB2bMail(DjfsUid uid, QuotaInfo quotaInfo, long fileCount) {
        JsonUtils.JsonObjectBuilder kwargsBuilder = JsonUtils.objectBuilder()
                .add("uid", uid.asString())
                .add("template", "b2b/space/finished")
                .add("data", spaceData(uid, quotaInfo, fileCount));

        CeleryJob celeryJob = CeleryJobUtils.create(SPACE_TASK_ID, kwargsBuilder.toMap());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));
    }
}
