package ru.yandex.chemodan.app.djfs.core.notification;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutorService;

import lombok.RequiredArgsConstructor;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.djfs.core.EventManager;
import ru.yandex.chemodan.app.djfs.core.album.Album;
import ru.yandex.chemodan.app.djfs.core.album.AlbumItem;
import ru.yandex.chemodan.app.djfs.core.filesystem.CopyActivity;
import ru.yandex.chemodan.app.djfs.core.filesystem.MoveActivity;
import ru.yandex.chemodan.app.djfs.core.filesystem.QuotaInfo;
import ru.yandex.chemodan.app.djfs.core.filesystem.event.PrivateFileCreatedEvent;
import ru.yandex.chemodan.app.djfs.core.filesystem.event.PrivateFolderCreatedEvent;
import ru.yandex.chemodan.app.djfs.core.filesystem.event.SharedFolderCreatedEvent;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.BlackboxUtils;
import ru.yandex.chemodan.app.djfs.core.operations.Operation;
import ru.yandex.chemodan.app.djfs.core.share.ShareInfo;
import ru.yandex.chemodan.app.djfs.core.share.event.UserKickedFromGroupEvent;
import ru.yandex.chemodan.app.djfs.core.share.event.UserLeftGroupEvent;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.CeleryJobUtils;
import ru.yandex.chemodan.app.djfs.core.util.DjfsAsyncTaskUtils;
import ru.yandex.chemodan.app.djfs.core.util.ExecutorServiceUtils;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.chemodan.app.djfs.core.util.JsonUtils;
import ru.yandex.chemodan.app.djfs.core.web.ConnectionIdHolder;
import ru.yandex.chemodan.queller.celery.job.CeleryJob;
import ru.yandex.chemodan.queller.worker.CeleryTaskManager;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonValue;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author eoshch
 */
@RequiredArgsConstructor
public class XivaPushGenerator {
    private static final Logger logger = LoggerFactory.getLogger(XivaPushGenerator.class);

    public static TaskId TASK_ID = new TaskId("mpfs.core.job_handlers.push.handle_push_notification");

    private final ExecutorService executorService;
    private final CeleryTaskManager celeryTaskManager;
    private final Blackbox2 blackbox2;

    private static long getDefaultOldVersion() {
        return InstantUtils.toVersion(Instant.now().minus(Duration.standardMinutes(1)));
    }

    private static JsonUtils.JsonObjectBuilder addConnectionId(JsonUtils.JsonObjectBuilder builder) {
        Option<String> connectionIdO = ConnectionIdHolder.getO();
        if (connectionIdO.isPresent()) {
            builder.add("connection_id", connectionIdO.get());
        }
        return builder;
    }

    private static JsonObject xivaData(DjfsResource resource, String op) {
        JsonUtils.JsonObjectBuilder result = JsonUtils.objectBuilder()
                .add("key", resource.getPath().getPath())
                .add("fid", resource.getFileId().map(DjfsFileId::getValue).getOrElse(""))
                .add("op", op);

        if (resource instanceof FileDjfsResource) {
            result.add("resource_type", "file")
                    .add("md5", ((FileDjfsResource) resource).getMd5())
                    .add("sha256", ((FileDjfsResource) resource).getSha256())
                    .add("size", ((FileDjfsResource) resource).getSize());

        } else {
            result.add("resource_type", "dir");
        }

        return result.toJsonObject();
    }

    private static MapF<String, JsonValue> diffKwargs(DjfsUid uid, Option<ShareInfo> shareInfo,
            DjfsResource resource, MapF<DjfsUid, Long> oldVersions)
    {
        return diffKwargs(uid, shareInfo, Cf.list(Tuple2.tuple(resource, XivaOpType.NEW)), oldVersions);
    }

    private static MapF<String, JsonValue> diffKwargs(DjfsUid uid, Option<ShareInfo> shareInfo,
            ListF<Tuple2<DjfsResource, XivaOpType>> resourcesWithType, MapF<DjfsUid, Long> oldVersions)
    {
        long defaultOldVersion = getDefaultOldVersion();
        long newVersion = resourcesWithType.map(Tuple2::get1).flatMap(x -> x.getVersion()).max();

        JsonUtils.JsonObjectBuilder kwargsBuilder = JsonUtils.objectBuilder()
                .add("new_version", Long.toString(newVersion))
                .addArray("xiva_data", resourcesWithType.map(x -> xivaData(x._1, x._2.value)))
                .add("class", "diff");

        if (shareInfo.isPresent()) {
            kwargsBuilder
                    .add("committer", uid.asString())
                    .add("action_name", "diff_group")
                    .add("gid", shareInfo.get().getGroupId())
                    .add("old_version", shareInfo.get().allUids().stream()
                            .map(x -> Tuple2.tuple(x.asString(), oldVersions.getOrElse(x, defaultOldVersion)))
                            .collect(JsonUtils::objectBuilder, (x, y) -> x.add(y), (x, y) -> x.addAll(y))
                            .toJsonObject());
        } else {
            kwargsBuilder
                    .add("uid", uid.asString())
                    .add("action_name", "diff")
                    .add("old_version", oldVersions.getOrElse(uid, defaultOldVersion));
        }

        return addConnectionId(kwargsBuilder).toMap();
    }

    @EventManager.EventHandler
    public void onPrivateFolderCreated(PrivateFolderCreatedEvent event) {
        if (!event.folder.getPath().getArea().isXivaPushNotificationSent()) {
            return;
        }

        MapF<DjfsUid, Long> oldVersions = event.oldVersion.isPresent()
                                          ? Cf.map(event.folder.getUid(), event.oldVersion.get()) : Cf.map();
        MapF<String, JsonValue> kwargs = diffKwargs(event.folder.getUid(), Option.empty(), event.folder,
                oldVersions);

        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, kwargs);
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));

        logger.info("Push sent onPrivateFolderCreated: " + kwargs);
    }

    @EventManager.EventHandler
    public void onSharedFolderCreated(SharedFolderCreatedEvent event) {
        if (!event.folder.getPath().getArea().isXivaPushNotificationSent()) {
            return;
        }

        MapF<String, JsonValue> kwargs = diffKwargs(event.actor, Option.of(event.shareInfo), event.folder,
                event.oldVersions.filterValues(x -> x.isPresent()).mapValues(x -> x.get()));

        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, kwargs);
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));

        logger.info("Push sent onSharedFolderCreated: " + kwargs);
    }

    @EventManager.EventHandler
    public void onPrivateFileCreated(PrivateFileCreatedEvent event) {
        if (!event.getFile().getPath().getArea().isXivaPushNotificationSent()) {
            return;
        }

        MapF<DjfsUid, Long> oldVersions = event.getOldVersion().isPresent()
                                          ? Cf.map(event.getFile().getUid(), event.getOldVersion().get()) : Cf.map();
        MapF<String, JsonValue> kwargs = diffKwargs(event.getFile().getUid(), Option.empty(), event.getFile(),
                oldVersions);

        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, kwargs);
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));

        logger.info("Push sent onPrivateFileCreated: " + kwargs);
    }

    private static MapF<String, JsonValue> buildShareLeaveEventPush(DjfsUid uid, ShareInfo shareInfo, String type,
            String klass, String forWhom, String connectionId, boolean addFolderName,
            Option<Tuple2<DjfsUid, String>> userData)
    {
        List<JsonValue> values = new ArrayList<>();

        JsonUtils.JsonObjectBuilder folderParameters = JsonUtils.objectBuilder()
                .add("path", shareInfo.getRootPath(uid).map(x -> x.getPath() + "/").getOrElse(""))
                .add("gid", shareInfo.getGroupId());
        if (addFolderName) {
            folderParameters.add("name", shareInfo.getRootPath(uid).map(DjfsResourcePath::getName).getOrElse(""));
        }

        values.add(JsonUtils.objectBuilder()
                .add("tag", "folder")
                .add("value", "")
                .add("parameters", folderParameters.toJsonObject())
                .toJsonObject());

        if (userData.isPresent()) {
            values.add(JsonUtils.objectBuilder()
                    .add("tag", "user")
                    .add("value", "")
                    .add("parameters", JsonUtils.objectBuilder()
                            .add("uid", userData.get()._1.asString())
                            .add("name", userData.get()._2)
                            .toJsonObject())
                    .toJsonObject());
        }

        return JsonUtils.objectBuilder()
                .add("new_version", 1)
                .add("uid", uid.asString())
                .add("xiva_data", JsonUtils.objectBuilder()
                        .add("values", JsonUtils.array(values))
                        .add("root", JsonUtils.objectBuilder()
                                .add("tag", "share")
                                .add("parameters", JsonUtils.objectBuilder()
                                        .add("type", type)
                                        .add("for", forWhom)
                                        .toJsonObject())
                                .toJsonObject())
                        .toJsonObject())
                .add("operation", "action")
                .add("connection_id", connectionId)
                .add("class", klass)
                .toMap();
    }

    private void handleShareLeaveEvent(DjfsUid uid, ShareInfo shareInfo, String type) {
        String klass = "share_" + type;

        MapF<String, JsonValue> kwargs1 = buildShareLeaveEventPush(uid, shareInfo, type, klass,
                "actor", ConnectionIdHolder.getO().getOrElse(""), false, Option.empty());
        CeleryJob celeryJob1 = CeleryJobUtils.create(TASK_ID, kwargs1);
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob1));

        logger.info("Push sent handleShareLeaveEvent: " + kwargs1);

        BlackboxCorrectResponse user = BlackboxUtils.getBlackboxUserInfo(uid, blackbox2);
        Tuple2<DjfsUid, String> userData = Tuple2.tuple(uid, user.getDisplayName().get().getPublicName().getOrElse(""));

        // push for owner
        MapF<String, JsonValue> kwargs2 = buildShareLeaveEventPush(shareInfo.getOwnerUid(), shareInfo, type, klass,
                "owner", "", true, Option.of(userData));
        CeleryJob celeryJob2 = CeleryJobUtils.create(TASK_ID, kwargs2);
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob2));

        logger.info("Push sent handleShareLeaveEvent: " + kwargs2);

        // push for other participants
        for (DjfsUid other : shareInfo.allUids()) {
            if (Objects.equals(other, shareInfo.getOwnerUid()) || Objects.equals(other, uid)) {
                continue;
            }

            MapF<String, JsonValue> kwargs3 = buildShareLeaveEventPush(other, shareInfo, type, klass,
                    "user", "", true, Option.of(userData));
            CeleryJob celeryJob3 = CeleryJobUtils.create(TASK_ID, kwargs3);
            ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob3));

            logger.info("Push sent handleShareLeaveEvent: " + kwargs3);
        }
    }

    @EventManager.EventHandler
    public void handle(UserKickedFromGroupEvent event) {
        handleShareLeaveEvent(event.getUid(), event.getShareInfo(), "user_was_banned");
    }

    @EventManager.EventHandler
    public void handle(UserLeftGroupEvent event) {
        handleShareLeaveEvent(event.getUid(), event.getShareInfo(), "user_has_left");
    }

    public void sendPush(CopyActivity activity, MapF<DjfsUid, Long> oldVersions) {
        if (!activity.getDestination().getPath().getArea().isXivaPushNotificationSent()) {
            return;
        }

        DjfsUid uid = activity.getPrincipal().getUidO().getOrElse(activity.getDestination().getUid());
        MapF<String, JsonValue> kwargs = diffKwargs(uid, activity.getDestinationShareInfo(), activity.getDestination(),
                oldVersions);

        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, kwargs);
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));

        logger.info("Push sent sendPush: " + kwargs);
    }

    public void sendPush(MoveActivity activity, MapF<DjfsUid, Long> oldVersions, ListF<DjfsResource> children) {
        if (!activity.getDestination().getPath().getArea().isXivaPushNotificationSent()) {
            return;
        }

        DjfsUid uid = activity.getPrincipal().getUidO().getOrElse(activity.getDestination().getUid());

        // TODO: support for shared folders with correct oldVersions

        Tuple2List<DjfsResource, XivaOpType> resourcesWithType = Tuple2List.fromPairs(
                activity.getSource(), XivaOpType.DELETED,
                activity.getDestination(), XivaOpType.NEW
        );
        for (DjfsResource child : children) {
            resourcesWithType.add(child, XivaOpType.NEW);
        }

        MapF<String, JsonValue> kwargs = diffKwargs(uid, Option.empty(), resourcesWithType.toList(), oldVersions);

        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, kwargs);
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));

        logger.info("Push sent sendPush: " + kwargs);
    }

    public void sendSpaceGroupPush(DjfsUid uid, ShareInfo shareInfo) {
        JsonUtils.JsonObjectBuilder kwargs = JsonUtils.objectBuilder()
                .add("committer", uid.asString())
                .add("gid", shareInfo.getGroupId())
                .add("new_version", "1")
                .add("operation", "action")
                .add("action_name", "space_group")
                .add("method", "copy_resource");

        addConnectionId(kwargs);

        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, kwargs.toMap());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));

        logger.info("Push sent sendSpaceGroupPush: " + kwargs.toMap());
    }

    public void sendSpaceIsLowPush(DjfsUid uid, QuotaInfo quotaInfo) {
        JsonUtils.JsonObjectBuilder kwargs = JsonUtils.objectBuilder()
                .add("uid", uid.asString())
                .add("new_version", "1")
                .add("class", "space_is_low")
                .add("xiva_data", JsonUtils.objectBuilder()
                        .add("root", JsonUtils.objectBuilder()
                                .add("tag", "space")
                                .add("parameters", JsonUtils.objectBuilder()
                                        .add("type", "is_low")
                                        .add("limit", quotaInfo.getLimit())
                                        .add("free", quotaInfo.getFree())
                                        .add("used", quotaInfo.getUsed())
                                        .toJsonObject())
                                .toJsonObject())
                        .add("values", JsonUtils.array())
                        .toJsonObject())
                .add("operation", "action")
                .add("action_name", "space");

        addConnectionId(kwargs);

        String deduplicationId = DjfsAsyncTaskUtils.activeUid("xiva_push_space__space_is_low_" + uid.asString());

        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, deduplicationId, kwargs.toMap());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));

        logger.info("Push sent sendSpaceIsLowPush: " + kwargs.toMap());
    }

    public void sendSpaceIsFullPush(DjfsUid uid, QuotaInfo quotaInfo) {
        JsonUtils.JsonObjectBuilder kwargs = JsonUtils.objectBuilder()
                .add("uid", uid.asString())
                .add("new_version", "1")
                .add("class", "space_is_full")
                .add("xiva_data", JsonUtils.objectBuilder()
                        .add("root", JsonUtils.objectBuilder()
                                .add("tag", "space")
                                .add("parameters", JsonUtils.objectBuilder()
                                        .add("type", "is_full")
                                        .add("limit", quotaInfo.getLimit())
                                        .add("free", quotaInfo.getFree())
                                        .add("used", quotaInfo.getUsed())
                                        .toJsonObject())
                                .toJsonObject())
                        .add("values", JsonUtils.array())
                        .toJsonObject())
                .add("operation", "action")
                .add("action_name", "space");

        addConnectionId(kwargs);

        String deduplicationId = DjfsAsyncTaskUtils.activeUid("xiva_push_space__space_is_full_" + uid.asString());

        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, deduplicationId, kwargs.toMap());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));

        logger.info("Push sent sendSpaceIsFullPush: " + kwargs.toMap());
    }

    public void sendOperationCreatedPush(DjfsUid uid, Operation operation) {
        JsonUtils.JsonObjectBuilder kwargs = JsonUtils.objectBuilder()
                .add("uid", uid.asString())
                .add("class", "operations")
                .add("xiva_data", JsonUtils.objectBuilder()
                        .add("root", JsonUtils.objectBuilder()
                                .add("tag", "operation")
                                .add("parameters", JsonUtils.objectBuilder()
                                        .add("oid", operation.getId())
                                        .add("type", "created")
                                        .toJsonObject())
                                .toJsonObject())
                        .add("values", JsonUtils.array())
                        .toJsonObject());

        addConnectionId(kwargs);

        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, kwargs.toMap());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));

        logger.info("Push sent sendOperationCreatedPush: " + kwargs.toMap());
    }

    /** A push event for mobile clients only */
    public void sendAlbumsDatabaseChangedPush(DjfsUid uid, long revision) {
        JsonUtils.JsonObjectBuilder kwargs = JsonUtils.objectBuilder()
                .add("uid", uid.asString())
                .add("class", "album_deltas_updated")
                .add("xiva_data", JsonUtils.objectBuilder()
                        .add("t", "album_deltas_updated")
                        .add("uid", uid.asString())
                        .add("revision", revision)
                .toJsonObject())
                .add("repack", JsonUtils.objectBuilder()
                        .add("other", JsonUtils.objectBuilder()
                                .add("repack_payload", JsonUtils.array(
                                        JsonUtils.value("t"),
                                        JsonUtils.value("uid"),
                                        JsonUtils.value("revision")))
                        .toJsonObject())
                .toJsonObject());

        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, kwargs.toMap());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));

        logger.info("Push sent sendAlbumsDatabaseChangedPush: " + kwargs.toMap());
    }

    public void sendAlbumItemRemovedPush(DjfsUid uid, Album album, AlbumItem item, boolean autoHandle) {
        /**
         * autoHandle - определяет, что пуш отправляется автоматической операцией (например, нотификацией
         * из Поиска. На данный признак ориентируется web-gui чтобы не показывать pop-ups для автоматических
         * операций.
         */
        JsonUtils.JsonObjectBuilder kwargs = JsonUtils.objectBuilder()
                .add("uid", uid.asString())
                .add("class", "albums")
                .add("xiva_data", JsonUtils.objectBuilder()
                        .add("root", JsonUtils.objectBuilder()
                                .add("tag", "album")
                                .add("parameters", JsonUtils.objectBuilder()
                                        .add("id", album.getId().toHexString())
                                        .add("type", "items_remove")
                                        .add("auto_handle", autoHandle ? 1 : 0)
                                        .add("public_key", album.getPublicKey().getOrElse(""))
                                        .toJsonObject())
                                .toJsonObject())
                        .add("values", JsonUtils.array(
                                JsonUtils.objectBuilder()
                                        .add("tag", "item")
                                        .add("parameters", JsonUtils.objectBuilder()
                                                .add("id", item.getId().toHexString())
                                                .toJsonObject())
                                        .toJsonObject())
                        )
                        .toJsonObject());

        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, kwargs.toMap());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));

        logger.info("Push sent sendAlbumItemRemovedPush: " + kwargs.toMap());
    }

    public void sendAlbumItemAppendPush(DjfsUid uid, Album album, AlbumItem item, boolean autoHandle) {
        /**
         * autoHandle - определяет, что пуш отправляется автоматической операцией (например, нотификацией
         * из Поиска. На данный признак ориентируется web-gui чтобы не показывать pop-ups для автоматических
         * операций.
         */
        JsonUtils.JsonObjectBuilder kwargs = JsonUtils.objectBuilder()
                .add("uid", uid.asString())
                .add("class", "albums")
                .add("xiva_data", JsonUtils.objectBuilder()
                        .add("root", JsonUtils.objectBuilder()
                                .add("tag", "album")
                                .add("parameters", JsonUtils.objectBuilder()
                                        .add("id", album.getId().toHexString())
                                        .add("type", "append_items")
                                        .add("auto_handle", autoHandle ? 1 : 0)
                                        .add("public_key", album.getPublicKey().getOrElse(""))
                                        .toJsonObject())
                                .toJsonObject())
                        .add("values", JsonUtils.array(
                                JsonUtils.objectBuilder()
                                        .add("tag", "item")
                                        .add("parameters", JsonUtils.objectBuilder()
                                                .add("id", item.getId().toHexString())
                                                .toJsonObject())
                                        .toJsonObject())
                        )
                        .toJsonObject());

        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, kwargs.toMap());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));

        logger.info("Push sent sendAlbumItemAppendPush: " + kwargs.toMap());
    }

    /** A push event for web client only */
    public void sendAlbumTitleChangedPush(DjfsUid uid, Album album ) {
        JsonUtils.JsonObjectBuilder kwargs = JsonUtils.objectBuilder()
                .add("uid", uid.asString())
                .add("class", "albums")
                .add("xiva_data", JsonUtils.objectBuilder()
                        .add("root", JsonUtils.objectBuilder()
                                .add("tag", "album")
                                .add("parameters", JsonUtils.objectBuilder()
                                        .add("id", album.getId().toHexString())
                                        .add("type", "title_change")
                                        .add("public_key", album.getPublicKey().getOrElse(""))
                                        .add("title", album.getTitle())
                                        .toJsonObject())
                                .toJsonObject())
                        .toJsonObject());

        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, kwargs.toMap());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));

        logger.info("Push sent sendAlbumTitleChangedPush: " + kwargs.toMap());
    }

    /** A push event for web client only */
    public void sendAlbumCoverChangedPush(DjfsUid uid, Album album) {
        JsonUtils.JsonObjectBuilder kwargs = JsonUtils.objectBuilder()
                .add("uid", uid.asString())
                .add("class", "albums")
                .add("xiva_data", JsonUtils.objectBuilder()
                        .add("root", JsonUtils.objectBuilder()
                                .add("tag", "album")
                                .add("parameters", JsonUtils.objectBuilder()
                                        .add("id", album.getId().toHexString())
                                        .add("type", "cover_change")
                                        .add("public_key", album.getPublicKey().getOrElse(""))
                                        .toJsonObject())
                                .toJsonObject())
                        .add("values", JsonUtils.array(
                                JsonUtils.objectBuilder()
                                        .add("tag", "cover")
                                        .add("parameters", JsonUtils.objectBuilder()
                                                .add("id", album.getCoverId().get().toHexString())
                                                .toJsonObject())
                                        .toJsonObject())
                        )
                        .toJsonObject());

        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, kwargs.toMap());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));

        logger.info("Push sent sendAlbumCoverChangedPush: " + kwargs.toMap());
    }

    public void sendAlbumCoverOffsetChangedPush(DjfsUid uid, Album album) {
        JsonUtils.JsonObjectBuilder kwargs = JsonUtils.objectBuilder()
                .add("uid", uid.asString())
                .add("class", "albums")
                .add("xiva_data", JsonUtils.objectBuilder()
                        .add("root", JsonUtils.objectBuilder()
                                .add("tag", "album")
                                .add("parameters", JsonUtils.objectBuilder()
                                        .add("id", album.getId().toHexString())
                                        .add("type", "cover_offset_change")
                                        .add("public_key", album.getPublicKey().getOrElse(""))
                                        .toJsonObject())
                                .toJsonObject())
                        .add("values", JsonUtils.array(
                                JsonUtils.objectBuilder()
                                        .add("tag", "cover")
                                        .add("parameters", JsonUtils.objectBuilder()
                                                .add("cover_offset_y", String.valueOf(album.getCoverOffsetY().getOrElse(0d)))
                                                .toJsonObject())
                                        .toJsonObject())
                        )
                        .toJsonObject());

        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, kwargs.toMap());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));

        logger.info("Push sent sendAlbumCoverOffsetChangedPush: " + kwargs.toMap());
    }

    public void sendAlbumPublicPush(DjfsUid uid, Album album) {
        String publicStatus = (album.isPublic()) ? "publish" : "unpublish";
        JsonUtils.JsonObjectBuilder kwargs = JsonUtils.objectBuilder()
                .add("uid", uid.asString())
                .add("class", "albums")
                .add("xiva_data", JsonUtils.objectBuilder()
                        .add("root", JsonUtils.objectBuilder()
                                .add("tag", "album")
                                .add("parameters", JsonUtils.objectBuilder()
                                        .add("id", album.getId().toHexString())
                                        .add("type", publicStatus)
                                        .add("public_key", album.getPublicKey().getOrElse(""))
                                        .add("is_public", String.valueOf(album.isPublic()))
                                        .toJsonObject())
                                .toJsonObject())
                        .toJsonObject());

        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, kwargs.toMap());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));

        logger.info("Push sent sendAlbumPublicPush: " + kwargs.toMap());
    }

    public void sendAlbumCreatedPush(DjfsUid uid, Album album) {
        JsonUtils.JsonObjectBuilder kwargs = JsonUtils.objectBuilder()
                .add("uid", uid.asString())
                .add("class", "albums")
                .add("xiva_data", JsonUtils.objectBuilder()
                        .add("root", JsonUtils.objectBuilder()
                                .add("tag", "album")
                                .add("parameters", JsonUtils.objectBuilder()
                                        .add("id", album.getId().toHexString())
                                        .add("type", "album_create")
                                        .add("public_key", album.getPublicKey().getOrElse(""))
                                        .toJsonObject())
                                .toJsonObject())
                        .toJsonObject());

        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, kwargs.toMap());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));

        logger.info("Push sent sendAlbumCreatedPush: " + kwargs.toMap());
    }

    /** A push event for web client only */
    public void sendAlbumRemovedPush(DjfsUid uid, Album album) {
        sendPush(XivaPushGeneratorAlbumUtils.albumRemoved(uid, album));

        logger.info("Push sent sendAlbumRemovedPush: {uid: " + uid.asString() + ", album_id: " + album.getId().toString() + "}");
    }

    public void sendPush(XivaData data) {
        CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, data.toCeleryKwargs());
        ExecutorServiceUtils.executeWithMdc(executorService, () -> celeryTaskManager.submit(celeryJob));
    }
}
