package ru.yandex.chemodan.app.djfs.core.operations;

import org.joda.time.DateTime;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.db.mongo.DjfsBenderFactory;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.chemodan.app.djfs.core.util.ZipUtils;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonValue;
import ru.yandex.commune.mongo3.MongoBenderParserSerializer;
import ru.yandex.commune.mongo3.bender.MongoId;
import ru.yandex.misc.bender.BenderParserSerializer;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;

/**
 * @author eoshch
 */
@BenderBindAllFields
public class MongoOperation {
    public static final MongoBenderParserSerializer<String, MongoOperation> B =
            DjfsBenderFactory.createForMongoSnakeCase(MongoOperation.class);

    @MongoId
    public String id;
    public String uid;
    public int state;

    // todo: бывает int и DateTime
    public DateTime dtime;

    @BenderPart( name = "uniq_id", strictName = true)
    public Option<String> uniq_id;
    public int mtime;
    public Option<String> md5 = Option.empty();
    // todo: может не быть?
    public long version;
    public byte[] data;

    public Operation to() {
        byte[] jsonData = ZipUtils.decompress(data);
        Data data = Data.B.getParser().parseJson(jsonData);
        return Operation.builder()
                .id(id)
                .uid(DjfsUid.cons(uid))
                .type(data.type)
                .subtype(data.subtype)
                .uniqueId(uniq_id.map(UuidUtils::fromHex))
                .state(Operation.State.R.fromValue(state))
                .version(version)
                .ctime(InstantUtils.fromSeconds(data.ctime))
                .mtime(InstantUtils.fromSeconds(mtime))
                .dtime(dtime.minusHours(3).toInstant())
                .jsonData((JsonObject) data.data)
                .build();
    }

    public static MongoOperation cons(Operation operation) {
        MongoOperation result = new MongoOperation();
        result.id = operation.getId();
        result.uid = operation.getUid().asString();
        result.state = operation.getState().value();
        result.uniq_id = operation.getUniqueId().map(UuidUtils::toHexString);
        result.dtime = operation.getDtime().toDateTime().plusHours(3);
        result.mtime = InstantUtils.toSeconds(operation.getMtime());
        result.md5 = Option.empty();
        result.version = operation.getVersion();
        Data data = new Data();
        data.id = operation.getId();
        data.ctime = InstantUtils.toSeconds(operation.getCtime());
        data.type = operation.getType();
        data.subtype = operation.getSubtype();
        data.data = operation.getJsonData();
        result.data = ZipUtils.compress(Data.B.getSerializer().serializeJson(data), 1);
        return result;
    }

    @BenderBindAllFields
    public static class Data {
        public String id;
        public String type;
        public String subtype;
        public int ctime;
        public JsonValue data;

        public static final BenderParserSerializer<Data> B = DjfsBenderFactory.createForJson(Data.class);
    }
}
