package ru.yandex.chemodan.app.djfs.core.operations;

import java.util.UUID;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import lombok.RequiredArgsConstructor;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.SafeCloseable;
import ru.yandex.chemodan.app.djfs.core.tasks.DelayDjfsCeleryOnetimeTaskException;
import ru.yandex.chemodan.app.djfs.core.tasks.DjfsCeleryOnetimeTaskLogger;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.ThreadLocalCacheUtil;
import ru.yandex.chemodan.queller.worker.CeleryOnetimeTask;
import ru.yandex.chemodan.queller.worker.CeleryOnetimeTaskParameters;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.json.JsonString;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author eoshch
 */
@RequiredArgsConstructor
public abstract class MpfsOperationHandler extends CeleryOnetimeTask {
    private static final Logger logger = LoggerFactory.getLogger(MpfsOperationHandler.class);
    private static final Logger errorLogger = LoggerFactory.getLogger("error");

    protected final OperationDao operationDao;

    private final Duration lockExpiry;
    private final Duration lockUpdateInterval;

    public MpfsOperationHandler(MpfsOperationHandlerContext mpfsOperationHandlerContext) {
        this.operationDao = mpfsOperationHandlerContext.operationDao;
        this.lockExpiry = mpfsOperationHandlerContext.mpfsOperationHandlerProperties.getLockExpiry();
        this.lockUpdateInterval = mpfsOperationHandlerContext.mpfsOperationHandlerProperties.getLockUpdateInterval();
        logger.info("MpfsOperationHandler initialized with "
                + mpfsOperationHandlerContext.mpfsOperationHandlerProperties.toString());
    }

    protected final void tryUpdateLock(Operation operation, UUID lockId) {
        if (!operationDao.tryAcquireOrRenewLockForExecution(operation.getUid(), operation.getId(), lockId,
                lockExpiry))
        {
            logger.info("did not acquire operation lock, delay");
            throw new DelayDjfsCeleryOnetimeTaskException();
        }
    }

    @Override
    protected final void execute(CeleryOnetimeTaskParameters parameters, ExecutionContext context) {
        DjfsCeleryOnetimeTaskLogger.Status taskStatus = DjfsCeleryOnetimeTaskLogger.Status.FAIL;
        UUID lockId = UUID.randomUUID();
        Option<Operation> operationO = Option.empty();
        Option<Operation.State> operationStateO = Option.empty();
        ScheduledExecutorService lockUpdater = Executors.newScheduledThreadPool(1);

        try (SafeCloseable ignored = ThreadLocalCacheUtil.withThreadLocalCache()) {
            try {
                DjfsUid uid = DjfsUid.cons(((JsonString) parameters.kwargs.getTs("uid")).getString());
                String oid = ((JsonString) parameters.kwargs.getTs("oid")).getString();

                operationO = operationDao.find(uid, oid);
                operationStateO = operationO.map(Operation::getState);

                Operation operation = operationO.getOrThrow(() -> new OperationNotFoundException(uid, oid));
                Operation.State state = operation.getState();
                if (state != Operation.State.WAITING && state != Operation.State.EXECUTING) {
                    logger.info("operation " + uid.asString() + ":" + operation.getId() + " is in state " + state.name()
                            + ", not executing");
                    taskStatus = DjfsCeleryOnetimeTaskLogger.Status.OK;
                    return;
                }

                AtomicBoolean terminated = new AtomicBoolean(false);

                tryUpdateLock(operation, lockId);
                // tryAcquireOrRenewLockForExecution sets state to executing
                operationStateO = Option.of(Operation.State.EXECUTING);

                lockUpdater.scheduleWithFixedDelay(() -> {
                    try {
                        tryUpdateLock(operation, lockId);
                    } catch (Exception e) {
                        terminated.set(true);
                    }
                }, lockUpdateInterval.getMillis(), lockUpdateInterval.getMillis(), TimeUnit.MILLISECONDS);

                Status status = handle(operation, terminated);
                if (status == Status.DONE) {
                    operationDao.changeState(uid, oid, Operation.State.EXECUTING, Operation.State.COMPLETED);
                    operationStateO = Option.of(Operation.State.COMPLETED);
                } else if (status == Status.DELAY) {
                    throw new DelayDjfsCeleryOnetimeTaskException();
                } else if (status == Status.FAIL) {
                    operationDao.changeState(uid, oid, Operation.State.EXECUTING, Operation.State.FAILED);
                    operationStateO = Option.of(Operation.State.FAILED);
                } else {
                    throw new NotImplementedException();
                }
                taskStatus = DjfsCeleryOnetimeTaskLogger.Status.OK;
            } catch (DelayDjfsCeleryOnetimeTaskException e) {
                // do not log exception as unhandled
                taskStatus = DjfsCeleryOnetimeTaskLogger.Status.DELAY;
                throw e;
            } catch (Exception e) {
                // taskStatus is FAIL already, no need to set
                errorLogger.error("mpfs operation unhandled: ", e);
                throw e;
            } finally {
                DjfsCeleryOnetimeTaskLogger.log(taskStatus, parameters, context, operationO, operationStateO);
                lockUpdater.shutdownNow();
                if (operationO.isPresent()) {
                    operationDao.releaseLock(operationO.get().getUid(), operationO.get().getId(), lockId);
                }
            }
        }
    }

    protected abstract Status handle(Operation operation, AtomicBoolean terminated);

    public enum Status {
        DONE, FAIL, DELAY
    }
}
