package ru.yandex.chemodan.app.djfs.core.operations;

import java.util.UUID;

import lombok.Builder;
import lombok.Value;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.db.DjfsUidSource;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.chemodan.util.bender.BenderUtils;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.misc.bender.BenderParserSerializer;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.enums.OrdinalIntEnum;
import ru.yandex.misc.enums.OrdinalIntEnumResolver;

/**
 * @author eoshch
 */
@Value
@Builder(toBuilder = true)
public class Operation implements DjfsUidSource {
    private String id;
    private DjfsUid uid;

    private String type;
    private String subtype;

    @Builder.Default
    private Option<UUID> uniqueId = Option.empty();

    private State state;
    private long version;
    private Instant ctime;
    private Instant mtime;
    private Instant dtime;

    private JsonObject jsonData;

    public <T> T getData(BenderParserSerializer<T> bender) {
        return BenderUtils.fromJsonObject(bender.getParser(), jsonData);
    }

    public boolean isFinished() {
        return state != State.WAITING && state != State.EXECUTING;
    }

    public Operation withUniqueIdFromUidAndTypeAndSubtype() {
        return toBuilder().uniqueId(Option.of(uniqueId(uid, type, subtype))).build();
    }

    public Operation withUniqueId(UUID uniqueId) {
        return toBuilder().uniqueId(Option.of(uniqueId)).build();
    }

    public static UUID uniqueId(DjfsUid uid, String type, String subtype) {
        return UuidUtils.md5AsUuid(uid.asString() + ":" + type + ":" + subtype);
    }

    public static Operation cons(DjfsUid uid, String type, String subtype) {
        return cons(uid, type, subtype, Option.empty(), JsonObject.empty());
    }

    public static Operation cons(DjfsUid uid, String type, String subtype, JsonObject data) {
        return cons(uid, type, subtype, Option.empty(), data);
    }

    public static <T> Operation cons(DjfsUid uid, String type, String subtype, T data,
            BenderParserSerializer<T> bender)
    {
        return cons(uid, type, subtype, Option.empty(), BenderUtils.toJsonObject(bender.getSerializer(), data));
    }

    public static Operation cons(DjfsUid uid, String type, String subtype, Option<UUID> uniqueId) {
        return cons(uid, type, subtype, uniqueId, JsonObject.empty());
    }

    public static Operation cons(DjfsUid uid, String type, String subtype, Option<UUID> uniqueId, JsonObject data) {
        Instant now = Instant.now();
        return Operation.builder()
                .id(UuidUtils.randomToHexString() + UuidUtils.randomToHexString())
                .uid(uid)
                .type(type)
                .subtype(subtype)
                .uniqueId(uniqueId)
                .jsonData(data)
                .state(State.WAITING)
                .version(InstantUtils.toVersion(now))
                .ctime(now)
                .mtime(now)
                .dtime(now)
                .build();
    }

    public enum State implements OrdinalIntEnum {
        WAITING(0),
        EXECUTING(1),
        FAILED(2),
        DONE(3),
        ABORTED(4),
        REJECTED(5),
        COMPLETED(6),
        ;

        private final int value;

        State(int value) {
            this.value = value;
        }

        @Override
        public int value() {
            return value;
        }

        public static final OrdinalIntEnumResolver<State> R = OrdinalIntEnumResolver.r(State.class);
    }

    @Value
    @BenderBindAllFields
    public static class ErrorData {
        private final int response;
        private final int code;
        private final String message;
    }
}
