package ru.yandex.chemodan.app.djfs.core.operations;

import java.util.Map;
import java.util.UUID;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.dao.support.DataAccessUtils;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDao;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.misc.bender.BenderParserSerializer;
import ru.yandex.misc.db.resultSet.ResultSetUtils;
import ru.yandex.misc.lang.CharsetUtils;

/**
 * @author eoshch
 */
public class PgOperationDao extends PgShardedDao implements OperationDao {
    private final static RowMapper<Operation> M = (rs, rowNum) -> Operation.builder()
            .id(CharsetUtils.decodeAscii(rs.getBytes("id")))
            .uid(DjfsUid.cons(rs.getLong("uid")))
            .type(rs.getString("type"))
            .subtype(rs.getString("subtype"))
            .uniqueId(Option.ofNullable(rs.getString("uniq_id")).map(UuidUtils::from))
            .state(Operation.State.R.fromValue(rs.getInt("state")))
            .version(rs.getLong("version"))
            .ctime(new Instant(rs.getTimestamp("ctime")))
            .mtime(new Instant(rs.getTimestamp("mtime")))
            .dtime(new Instant(rs.getTimestamp("dtime")))
            .jsonData(JsonObject.parseObject(rs.getString("data")))
            .build();

    public PgOperationDao(PgShardedDaoContext context) {
        super(context);
    }

    @Override
    public void insert(Operation operation) {
        String sql = collectStats(operation)
                + " INSERT INTO disk.operations (id, uid, ctime, dtime, mtime, state, version, type, subtype, md5, "
                + " uniq_id, data) "
                + " VALUES (:id, :uid, :ctime, :dtime, :mtime, :state, :version, :type, :subtype, :md5, "
                + " :uniq_id, :data::json)";

        Map<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "id", CharsetUtils.encodeAsciiToArray(operation.getId()),
                "uid", operation.getUid(),
                "ctime", operation.getCtime(),
                "dtime", operation.getDtime(),
                "mtime", operation.getMtime(),
                "state", operation.getState(),
                "version", operation.getVersion(),
                "type", operation.getType(),
                "subtype", operation.getSubtype(),
                "md5", null,
                "uniq_id", operation.getUniqueId().getOrNull(),
                "data", operation.getJsonData().serialize()
        ));

        jdbcTemplate(operation.getUid()).update(sql, parameters);
    }

    @Override
    public Option<Operation> find(DjfsUid uid, String id) {
        String sql = collectStats(uid) + " SELECT * FROM disk.operations WHERE id = :id";
        Map<String, Object> parameters = Cf.map("id", CharsetUtils.encodeAsciiToArray(id));
        return jdbcTemplate(uid).queryForOption(sql, M, parameters);
    }

    @Override
    public ListF<Operation> find(DjfsUid uid, String type, String subtype) {
        String sql = collectStats(uid)
                + " SELECT * FROM disk.operations WHERE uid = :uid AND type = :type AND subtype = :subtype";
        Map<String, Object> parameters = Cf.map(
                "uid", uid.asLong(),
                "type", type,
                "subtype", subtype);
        return jdbcTemplate(uid).query(sql, M, parameters);
    }

    @Override
    public long count(DjfsUid uid, ListF<Operation.State> states, ListF<String> notTypes) {
        String sql = "SELECT count(*) FROM disk.operations "
                + "WHERE uid = :uid AND state IN (:states) AND type NOT IN (:notTypes)";
        Map<String, Object> parameters = Cf.map(
                "uid", uid.asLong(),
                "states", states,
                "notTypes", notTypes
                );
        return DataAccessUtils.requiredSingleResult(
                jdbcTemplate(uid).query(sql, ResultSetUtils.defaultSingleColumnRowMapper(Long.class), parameters)
        );
    }

    @Override
    public long count(DjfsUid uid, Instant after) {
        String sql = "SELECT count(*) FROM disk.operations WHERE uid = :uid AND dtime > :after";
        Map<String, Object> parameters = Cf.map(
                "uid", uid.asLong(),
                "after", after);
        return DataAccessUtils.requiredSingleResult(
                jdbcTemplate(uid).query(sql, ResultSetUtils.defaultSingleColumnRowMapper(Long.class), parameters)
        );
    }

    @Override
    public boolean changeState(DjfsUid uid, String id, Operation.State from, Operation.State to) {
        String sql = collectStats(uid) + " UPDATE disk.operations SET state = :new_state "
                + " WHERE id = :id AND state = :old_state";
        Map<String, Object> parameters = Cf.map(
                "id", CharsetUtils.encodeAsciiToArray(id),
                "old_state", from.value(),
                "new_state", to.value());
        int updated = jdbcTemplate(uid).update(sql, parameters);
        return updated > 0;
    }

    @Override
    public boolean setDtime(DjfsUid uid, String id, Instant dtime) {
        String sql = collectStats(uid) + " UPDATE disk.operations SET dtime = :dtime WHERE id = :id";
        Map<String, Object> parameters = Cf.map(
                "id", CharsetUtils.encodeAsciiToArray(id),
                "dtime", dtime);
        int updated = jdbcTemplate(uid).update(sql, parameters);
        return updated > 0;
    }

    @Override
    public <T> void setData(DjfsUid uid, String id, T data, BenderParserSerializer<T> bender, Operation operation) {
        String sql = collectStats(uid) + " UPDATE disk.operations SET data = :data::json WHERE id = :id";
        Map<String, Object> parameters = Cf.map(
                "id", CharsetUtils.encodeAsciiToArray(id),
                "data", CharsetUtils.decodeUtf8(bender.getSerializer().serializeJson(data)));
        jdbcTemplate(uid).update(sql, parameters);
    }

    @Override
    public boolean tryAcquireOrRenewLockForExecution(DjfsUid uid, String id, UUID lockId, Duration lockExpiry) {
        Instant now = Instant.now();
        Instant threshold = now.minus(lockExpiry);

        String sql = collectStats(uid) + " UPDATE disk.operations "
                + " SET state = 1, lock_id = :lock_id, date_lock_acquired = :date_lock_acquired, dtime = :dtime "
                + " WHERE id = :id AND uid = :uid AND (state = 0 OR state = 1) "
                + "     AND ( lock_id IS NULL OR lock_id = :lock_id OR (lock_id IS NOT NULL AND date_lock_acquired < :threshold))";

        Map<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "id", CharsetUtils.encodeAsciiToArray(id),
                "uid", uid,
                "lock_id", lockId,
                "date_lock_acquired", now,
                "dtime", now,
                "threshold", threshold));
        int updated = jdbcTemplate(uid).update(sql, parameters);
        return updated > 0;
    }

    @Override
    public void releaseLock(DjfsUid uid, String id, UUID lockId) {
        Instant now = Instant.now();

        String sql = collectStats(uid) + " UPDATE disk.operations "
                + " SET lock_id = NULL, date_lock_acquired = NULL, dtime = :dtime "
                + " WHERE id = :id AND uid = :uid AND lock_id = :lock_id";

        Map<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "id", CharsetUtils.encodeAsciiToArray(id),
                "uid", uid,
                "lock_id", lockId,
                "dtime", now));
        jdbcTemplate(uid).update(sql, parameters);
    }
}
