package ru.yandex.chemodan.app.djfs.core.publication;

import java.time.temporal.ChronoUnit;

import com.mongodb.ReadPreference;
import lombok.RequiredArgsConstructor;
import org.joda.time.DateTime;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.filesystem.QuotaManager;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.Blockings;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.commune.dynproperties.DynamicProperty;

@RequiredArgsConstructor
public class BlockingsManager {

    private final QuotaManager quotaManager;
    private final MongoBlockingsDao blockingsDao;

    private final DynamicProperty<Long> overdraftRestrictionsThreshold = new DynamicProperty<>("djfs.public.overdraft-restrictions-threshold", 1073741824L);

    public ListF<FileDjfsResource> getBlockingsForFiles(ListF<FileDjfsResource> resources) {
        final Option<Long> downloadSpeedLimitEndTime = resources.firstO().filterMap(resource -> quotaManager.getDownloadSpeedLimitEndTime(resource.getUid()));
        if (downloadSpeedLimitEndTime.isPresent()) {
            final Option<MapF<String, Long>> simpleO = Option.of(Cf.map("public", downloadSpeedLimitEndTime.get()));
            return resources
                    .map(r -> r.toBuilder().blockings(new Blockings(simpleO, Option.empty(), Option.empty(), Option.of(true), "")).build());
        }
        final Option<Long> freeQuota = resources.firstO().map(resource -> quotaManager.getFree(resource.getUid(),
                Option.of(ReadPreference.secondaryPreferred())));
        if (freeQuota.isMatch(free -> free <= 0)) {
            final Option<MapF<String, Long>> simpleO =
                    Option.of(Cf.map("public", java.time.Instant.now().plus(1L, ChronoUnit.DAYS).getEpochSecond()));
            return resources
                    .map(r -> r.toBuilder().blockings(new Blockings(simpleO, Option.empty(), Option.empty(), Option.of(true), "")).build());
        }

        final MapF<String, Blockings> blockingsMap = blockingsDao.find(resources.map(FileDjfsResource::getHid), ReadPreference.secondaryPreferred()).map(blockings -> {
            Option<MapF<String, Long>> simpleO = getMapWithAuthAndPublicValues(blockings.getSimple());
            Option<MapF<String, Long>> videoO = getMapWithAuthAndPublicValues(blockings.getVideo());
            return new Blockings(simpleO, videoO, Option.empty(), blockings.getPb(), new String(blockings.getId()));
        }).toMapMappingToKey(Blockings::getHid);
        final ListF<FileDjfsResource> result = Cf.arrayList();
        for (FileDjfsResource r : resources) {
            if (blockingsMap.containsKeyTs(r.getHid())) {
                result.add(r.toBuilder().blockings(blockingsMap.getTs(r.getHid())).build());
            } else {
                result.add(r);
            }
        }
        return result;
    }

    public DjfsResource getBlockingsByHash(DjfsResource resource, String hash) {
        final ListF<String> temporaryWorkaround = Cf.list("HASH:" + hash,
                "HASH:" + hash + ":"); //todo: fix on AntiFO side
        final Option<Blockings> blockingsO = blockingsDao.find(temporaryWorkaround, ReadPreference.secondaryPreferred()).firstO().map(blockings -> {
            Option<MapF<String, Long>> folderO = getMapWithAuthAndPublicValues(blockings.getFolder());
            return new Blockings(Option.empty(), Option.empty(), folderO, blockings.getPb(), new String(blockings.getId()));
        });
        return blockingsO.map(blockingsFound -> {
            final Blockings blockingsMerged = resource.getBlockings().map(blockings -> blockings.merge(blockingsFound))
                    .getOrElse(blockingsFound);
            return resource.toBuilder().blockings(blockingsMerged).build();
        }).getOrElse(resource);
    }

    private Option<MapF<String, Long>> getMapWithAuthAndPublicValues(MapF<String, DateTime> blockingsMapWithTime) {
        final Option<Long> authO = blockingsMapWithTime.getO("a").map(time -> time.getMillis() / 1000);
        final Option<Long> pubO = blockingsMapWithTime.getO("p").map(time -> time.getMillis() / 1000);
        final MapF<String, Long> map = Cf.hashMap();
        authO.map(auth -> map.put("auth", auth));
        pubO.map(pub -> map.put("public", pub));
        return Option.when(map.isNotEmpty(), map);
    }

    public boolean isInOverdraftForRestrictions(DjfsUid uid) {
        final long overdraft = -quotaManager.getFree(uid, Option.empty());
        return overdraft >= overdraftRestrictionsThreshold.get();
    }
}
