package ru.yandex.chemodan.app.djfs.core.publication;

import java.util.Map;

import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDao;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.db.pg.ResultSetUtils;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsFileId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.misc.codec.Hex;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author eoshch
 */
public class PgLinkDataDao extends PgShardedDao implements LinkDataDao {
    private final static RowMapper<LinkData> M = (rs, rowNum) -> LinkData.builder()
            .id(rs.getString("id"))
            .uid(DjfsUid.cons(rs.getLong("uid")))
            .resourceIdUid(ResultSetUtils.getLongO(rs, "resource_id_uid").map(riu ->  DjfsUid.cons(riu)))
            .targetPath(ResultSetUtils.getStringO(rs, "target").map(DjfsResourcePath::cons))
            .linkDataPath(ResultSetUtils.getStringO(rs, "path").getOrNull())
            .fileId(ResultSetUtils.getStringO(rs, "file_id").map(s-> StringUtils.stripStart(s, "\\x")).map(DjfsFileId::cons))
            .build();

    public PgLinkDataDao(PgShardedDaoContext context) {
        super(context);
    }

    @Override
    public void deleteAll(DjfsUid uid) {
        String sql = collectStats(uid) + " DELETE FROM disk.link_data WHERE uid = :uid";
        jdbcTemplate(uid).update(sql, Cf.map("uid", uid));
    }

    @Override
    public ListF<LinkData> find(DjfsUid uid, ListF<DjfsResourcePath> targetPaths) {
        if (targetPaths.isEmpty()) {
            return Cf.list();
        }
        String sql = collectStats(uid) + " SELECT * FROM disk.link_data WHERE uid = :uid AND target IN (:paths)";
        MapF<String, Object> parameters = Tuple2List.fromPairs(
                "uid", uid,
                "paths", targetPaths.map(DjfsResourcePath::getFullPath)
        ).toMap();

        return jdbcTemplate(uid).query(sql, M, parameters);
    }

    @Override
    public Option<LinkData> findBySymlink(DjfsUid uid, String path) {
        String sql = collectStats(uid) + " SELECT * FROM disk.link_data WHERE id = :id AND uid = :uid and date_deleted is null";
        MapF<String, Object> parameters = Tuple2List.fromPairs(
                "id", UuidUtils.md5AsUuid(uid+":"+path),
                "uid", uid
        ).toMap();

        String sqlByPath = collectStats(uid) + " SELECT * FROM disk.link_data WHERE uid = :uid AND path = (:path) and date_deleted is null";
        MapF<String, Object> parametersByPath = Tuple2List.fromPairs(
                "uid", uid,
                "path", path
        ).toMap();

        return jdbcTemplate(uid).query(sql, M, parameters).firstO()
                .orElse(() -> jdbcTemplate(uid).query(sqlByPath, M, parametersByPath).firstO());
    }

    @Override
    public void insert(LinkData linkData) {
        String sql = collectStats(linkData.getUid())
                + " INSERT INTO disk.link_data (id, uid, path, version, user_ip, public_uid, target, type, parent, "
                + " date_created, date_modified, date_deleted, file_id, resource_id_uid) VALUES (:id, :uid, :path, "
                + " :version, :user_ip, :public_uid, :target, :type::disk.resource_type, :parent, :date_created,"
                + " :date_modified, :date_deleted, :file_id, :resource_id_uid)";

        Map<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "id", UuidUtils.fromHex(linkData.getId()),
                "uid", linkData.getUid(),
                "path", linkData.getLinkDataPath(),
                "version", linkData.getVersion().getOrNull(),
                "user_ip", linkData.getUserIp().getOrNull(),
                "public_uid", linkData.getPublicUid().getOrNull(),
                "target", linkData.getTargetPath().map(DjfsResourcePath::getFullPath).getOrNull(),
                "type", linkData.getType(),
                "parent", linkData.getParentId().map(UuidUtils::fromHex).getOrNull(),
                "date_created", linkData.getDateCreated().getOrNull(),
                "date_modified", linkData.getDateModified().getOrNull(),
                "date_deleted", linkData.getDateDeleted().getOrNull(),
                "file_id", linkData.getResourceId().map(DjfsResourceId::getFileId).map(DjfsFileId::getValue).map(Hex::decode).getOrNull(),
                "resource_id_uid", linkData.getResourceId().map(DjfsResourceId::getUid).getOrNull()
        ));

        jdbcTemplate(linkData.getUid()).update(sql, parameters);
    }
}
