package ru.yandex.chemodan.app.djfs.core.publication;

import java.util.List;

import com.mongodb.ReadPreference;
import lombok.RequiredArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.djfs.core.DjfsException;
import ru.yandex.chemodan.app.djfs.core.LegacyMpfsAes;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsPrincipal;
import ru.yandex.chemodan.app.djfs.core.filesystem.Filesystem;
import ru.yandex.chemodan.app.djfs.core.filesystem.JsonSupportData;
import ru.yandex.chemodan.app.djfs.core.filesystem.SupportComment;
import ru.yandex.chemodan.app.djfs.core.filesystem.SupportDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.DjfsNotImplementedException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.OverdraftUserPublicLinkException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.ResourceBlockedException;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.ResourceNotFoundException;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.AntiVirusScanStatus;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DefaultFolder;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceId;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.exception.InvalidDjfsResourcePathException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyBadPathException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyPathErrorException;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.BlackboxUtils;
import ru.yandex.chemodan.app.djfs.core.legacy.formatting.UserPojo;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserDao;
import ru.yandex.chemodan.app.djfs.core.user.UserData;
import ru.yandex.chemodan.util.BleedingEdge;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxDbFields;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxDisplayName;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;

@RequiredArgsConstructor
public class PublicationManager {

    public static final SetF<Long> UNLIMITED_USERS_WHITE_LIST =
            Cf.set(222885258L, 222885028L, 222529628L, 44214498L, 134834766L, 221767951L,
                    175791915L, 175792789L, 175792410L, 175793063L, 215949621L, 199346514L,
                    210824649L, 213823174L, 217981775L, 221186076L, 221187854L, 207069215L,
                    170453517L, 185828634L, 271214600L, 566943057L, 566942851L, 566937221L,
                    566938106L, 616153853L, 1130000025984582L, 383946176L, 48493L, 1015103532L,
                    996716292L, 36841369L, 1057234642L, 653781586L, 424812249L);

    private final LegacyMpfsAes legacyMpfsAes;
    private final LinkDataDao linkDataDao;
    private final Filesystem filesystem;
    private final Blackbox2 blackbox;
    private final UserDao userDao;
    private final SupportDao supportDao;
    private final BlockingsManager blockingsManager;
    private final BleedingEdge bleedingEdge;

    public PublicAddress getPublicAddress(String hash) {
        if (StringUtils.isBlank(hash)) {
            throw new LegacyPathErrorException(new InvalidDjfsResourcePathException("Empty hash"));
        }
        final String publicAddressHash = StringUtils.substringBefore(hash, ":");
        final String publicAddressPath = StringUtils.substringAfter(hash, ":");
        final String symlinkAddr = legacyMpfsAes.decrypt(publicAddressHash);
        return new PublicAddress(publicAddressHash, symlinkAddr, publicAddressPath);
    }

    public DjfsResource getPublicResource(String accessUid, PublicAddress publicAddress)
    {
        LinkData linkData = getSymlinkResource(publicAddress.getDecodedHash())
                .getOrThrow(() -> new ResourceNotFoundException(publicAddress.getDecodedHash()));
        final UserData ownerUser = userDao.find(linkData.getUid())
                .getOrThrow(() -> new ResourceNotFoundException(publicAddress.getDecodedHash()));
        if (!ownerUser.isPg()) {
            throw new DjfsNotImplementedException();
        }
        final DjfsResource resourceFromHashPart = findResourceFromHashPart(linkData, ownerUser);
        final DjfsUid resoureUid = resourceFromHashPart.getUid();
        publicAddress.setUser(linkData.getUid().asLong() == resoureUid.asLong() ? ownerUser : userDao.find(resoureUid)
                .getOrThrow(() -> new ResourceNotFoundException(resourceFromHashPart.getPath())));
        if (!publicAddress.getUser().isPg()) {
            throw new DjfsNotImplementedException();
        }
        checkUserNotBlockedInPassport(resoureUid);
        if (isYaTeamSubtree(resourceFromHashPart)) {
            checkYaTeamSubtreeAccess(accessUid, publicAddress.getUser(),
                    () -> new ResourceNotFoundException(resourceFromHashPart.getPath()));
        }
        checkResourceBlocked(resourceFromHashPart);
        checkResourcePublic(resourceFromHashPart);
        checkUserInOverdraft(resoureUid);
        publicAddress.setFullPath(resourceFromHashPart.getPath().suffix(publicAddress.getPath()));
        checkIfParentFoldersBlocked(publicAddress.getFullPath(), publicAddress.getUser());
        DjfsResource resource = resourceFromHashPart;
        if (StringUtils.isNotBlank(publicAddress.getPath())) {
            resource = filesystem.find(DjfsPrincipal.cons(publicAddress.getUser()), publicAddress.getFullPath(), Option.of(ReadPreference.secondaryPreferred()))
                    .getOrThrow(() -> new ResourceNotFoundException(publicAddress.getFullPath()));
            checkResourceBlocked(resource);
        }
        return resource;
    }

    public boolean isYaTeamSubtree(DjfsResource resource) {
        return DefaultFolder.YATEAMNDA.representations().exists(resource.getPath().getPath()::startsWith);
    }

    public DjfsResource.Builder updatePathAndShortUrl(PublicAddress publicAddress, DjfsResource resource) {
        final String oldPath = StringUtils.substringBeforeLast(resource.getPath().getPath(), publicAddress.getPath());
        Option<String> name = restoreOriginalNameForAttachAndNarodResources(resource, publicAddress.getFullPath().getArea())
                .orElse(Option.of(publicAddress.getFullPath().getName()));
        final DjfsResourcePath newPath =
                resource.getPath().changeToPublicHash(oldPath, publicAddress.getEncodedHash(), resource.getUid(), name);
        DjfsResource.Builder resourceBuilder = resource.toBuilder()
                .path(newPath);
        if (isYaTeamSubtree(resource)) {
            resourceBuilder.shortUrl(createNdaResourceUrl(publicAddress.getEncodedHash()));
        }
        return resourceBuilder;
    }

    public String createNdaResourceUrl(String encodedHash) {
        final ListF<String> urlParts = Cf.list(encodedHash.split("/")).map(UrlUtils::urlEncode);
        return "https://disk.yandex.ru/public/nda/?hash=" + StringUtils.join(urlParts, "/");
    }

    public ListF<DjfsResource> processChildren(List<DjfsResource> children, SetF<DjfsResource> cleanedChildren,
            int pageBlockedItemsNum, DjfsResource resourceWithBlockings)
    {
        final ListF<DjfsResource> childrenForBlockings = Cf.arrayList();
        for (DjfsResource child : children) {
            if (cleanedChildren.containsTs(child)) {
                Option<String> originalNameO = restoreOriginalNameForAttachAndNarodResources(child, child.getPath().getArea());
                final DjfsResourcePath childNewPath = resourceWithBlockings.getPath().suffix("/" + originalNameO.getOrElse(child.getPath().getName()));
                DjfsResource.Builder childBuilder = child.toBuilder()
                        .pageBlockedItemsNum(pageBlockedItemsNum)
                        .path(childNewPath);
                childrenForBlockings.add(childBuilder.build());
            }
        }
        return checkBlockings(childrenForBlockings);
    }

    private Option<String> restoreOriginalNameForAttachAndNarodResources(DjfsResource resource, DjfsResourceArea area)
    {
        if (Cf.set(DjfsResourceArea.ATTACH, DjfsResourceArea.NAROD).containsTs(area)) {
            return resource.getExternalProperties().getO("original_name");
        }
        return Option.empty();
    }

    private Option<LinkData> getSymlinkResource(String symlinkAddr) {
        final String[] s = StringUtils.split(symlinkAddr, ':');
        if (s.length < 2 || !StringUtils.isNumericArabic(s[0])) {
            throw new LegacyBadPathException(new InvalidDjfsResourcePathException(new StringBuilder("Wrong symlink: ").append(symlinkAddr).toString()));
        }
        final String uidString = s[0];
        final String path = "/" + s[1];
        return linkDataDao.findBySymlink(DjfsUid.cons(uidString), path);
    }

    private DjfsResource findResourceFromHashPart(LinkData linkData, UserData ownerUser) {
        if (linkData.getFileId().isPresent()) {
            final DjfsPrincipal principal = linkData.getResourceIdUid().map(DjfsPrincipal::cons).getOrElse(DjfsPrincipal.cons(ownerUser));
            final DjfsUid uid = linkData.getResourceIdUid().getOrElse(linkData.getUid());
            final DjfsResourceId resourceId = DjfsResourceId.cons(uid, linkData.getFileId().get());
            return filesystem.find(principal, resourceId, Option.of(ReadPreference.secondaryPreferred()))
                    .filter(resource -> !Cf.set(DjfsResourceArea.HIDDEN, DjfsResourceArea.TRASH)
                            .containsTs(resource.getPath().getArea()))
                    .firstO().getOrThrow(() -> new ResourceNotFoundException(""));
        } else {
            return linkData.getTargetPath().filterMap(filesystem::findWithoutCheckingPermission)
                    .getOrThrow(() -> new ResourceNotFoundException(""));
        }
    }

    private void checkUserNotBlockedInPassport(DjfsUid uid) {
        BlackboxUtils.checkAvailable(uid, blackbox, () -> new ResourceNotFoundException(""));
    }

    private  <E extends DjfsException> void checkYaTeamSubtreeAccess(String accessUidString, UserData resourceUser,
            Function0<E> e)
    {
        if (StringUtils.isNotBlank(accessUidString) && !StringUtils.equals(accessUidString, "0")) {
            final DjfsUid accessUid = DjfsUid.cons(accessUidString);
            final Option<UserData> accessUser = userDao.find(accessUid);
            if (!userHasNdaRights(resourceUser) || !accessUser.isPresent() || !userHasNdaRights(accessUser.get())) {
                throw e.apply();
            }
        } else {
            throw e.apply();
        }
    }

    public boolean userHasNdaRights(UserData user) {
        return user.getYateamUid().isPresent() && BlackboxUtils.is2FAEnabled(user.getUid(), blackbox);
    }


    private void checkResourceBlocked(DjfsResource resource) {
        if (resource.isBlocked()) {
            throw new ResourceNotFoundException(resource.getPath());
        }
    }

    private void checkResourcePublic(DjfsResource resource) {
        if (!resource.isFullyPublic()) {
            throw new ResourceNotFoundException(resource.getPath());
        }
    }

    private void checkUserInOverdraft(DjfsUid uid) {
        final boolean overdraftFlag = blockingsManager.isInOverdraftForRestrictions(uid);
        if (bleedingEdge.isOnBleedingEdge(uid.asPassportUid()) && overdraftFlag) {
            throw new OverdraftUserPublicLinkException();
        }
    }

    private void checkIfParentFoldersBlocked(DjfsResourcePath path, UserData ownerUser) {
        final ListF<DjfsResource> parentResources =
                filesystem.findByPaths(DjfsPrincipal.cons(ownerUser), path.getAllParents().filter(parent -> !parent.isRoot()), Option.of(ReadPreference.secondaryPreferred()));
        for (DjfsResource resource : parentResources) {
            if (resource.isBlocked()) {
                throw new ResourceBlockedException(Option.empty());
            }
        }
    }

    public int countBlockedAndInfectedChildren(List<DjfsResource> children, SetF<DjfsResource> cleanedChildren)
    {
        int pageBlockedItemsNum = 0;
        for (DjfsResource child : children) {
            final boolean isInfected = child instanceof FileDjfsResource && ((FileDjfsResource) child)
                    .getAntiVirusScanStatus().isSome(AntiVirusScanStatus.INFECTED);
            if (child.isBlocked() || isInfected) {
                pageBlockedItemsNum++;
            } else {
                cleanedChildren.add(child);
            }
        }
        return pageBlockedItemsNum;
    }

    public ListF<DjfsResource> checkBlockings(ListF<DjfsResource> resources) {
        final boolean uidInWhiteList = resources.firstO().isMatch(r -> UNLIMITED_USERS_WHITE_LIST.containsTs(r.getUid().asLong()));
        if (uidInWhiteList) {
            return resources;
        }
        final ListF<DjfsResource> folders = resources.filter(r -> r instanceof FolderDjfsResource);
        final ListF<FileDjfsResource> files = resources.filter(r -> r instanceof FileDjfsResource).map(r -> (FileDjfsResource)r);
        final ListF<FileDjfsResource> filesWithBlockings = blockingsManager.getBlockingsForFiles(files);
        return folders.plus(filesWithBlockings);
    }

    public DjfsResource checkBlockingsByHash(DjfsResource resource, String hash) {
        return blockingsManager.getBlockingsByHash(resource, hash);
    }

    public UserPojo constructUserPojo(UserData ownerUser) {
        final BlackboxCorrectResponse ownerUserInfo = BlackboxUtils.getBlackboxUserInfo(ownerUser.getUid(), blackbox);
        final String login = ownerUserInfo.getLogin().getOrElse("");
        final String uid = ownerUser.getUid().asString();
        final String publicName = ownerUserInfo.getDisplayName().filterMap(BlackboxDisplayName::getPublicName).getOrElse("");
        final String displayName = ownerUserInfo.getDisplayName().map(BlackboxDisplayName::getName).getOrElse("");

        final Option<String> firstName =
                ownerUserInfo.getDbFields().getO(BlackboxDbFields.FIRSTNAME).filterNot(String::isEmpty);
        final Option<String> lastName =
                ownerUserInfo.getDbFields().getO(BlackboxDbFields.LASTNAME).filterNot(String::isEmpty);

        final String username = firstName.isPresent() && lastName.isPresent()
                                ? firstName.get() + " " + lastName.get()
                                : displayName;

        final String locale = ownerUserInfo.getDbFields().getTs(BlackboxDbFields.LANG);
        final boolean isPaid = ownerUser.isPaid();
        final boolean isAds = !isPaid && !ownerUser.isB2b();
        return new UserPojo(username, publicName, displayName, uid, locale, login, isPaid ? 1L : 0L, isAds ? 1L : 0L);
    }

    public void fillErrorData(ResourceNotFoundException ex, String hash)
    {
        final Option<SupportComment> supportComments = supportDao.find(hash);
        final Option<JsonSupportData> data = supportComments.map(SupportComment::getData);
        if (data.isPresent()) {
            final String link = data.get().getExtLink().getOrElse("");
            final String view = data.get().getView().getOrElse("");
            throw new ResourceBlockedException(Option.of(Cf.map("link", link, "view", view)));
        } else {
            throw ex;
        }
    }
}
