package ru.yandex.chemodan.app.djfs.core.share;

import com.mongodb.MongoClient;
import com.mongodb.ReadPreference;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.Updates;
import org.bson.BsonDocument;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.djfs.core.DjfsException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyUserIsReadOnlyException;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.commune.mongo3.MongoCollectionX;
import ru.yandex.misc.cache.tl.TlCache;

/**
 * @author eoshch
 */
public class MongoGroupDao implements GroupDao {
    private final MongoCollectionX<String, MongoGroup> collection;
    private final Function0<Boolean> isReadonly;

    public MongoGroupDao(MongoClient mongoClient, Function0<Boolean> isReadonly) {
        this.collection = new MongoCollectionX<>(
                mongoClient.getDatabase("groups").getCollection("groups", BsonDocument.class),
                MongoGroup.B);
        this.isReadonly = isReadonly;
    }

    private String getCacheKey(DjfsUid uid) {
        return "mongo group dao uid" + uid.asString();
    }

    private String getCacheKey(String groupId) {
        return "mongo group dao groupId " + groupId;
    }

    public Option<Group> find(String id) {
        return find(id, Option.of(ReadPreference.primary()));
    }

    public Option<Group> find(String id, Option<ReadPreference> readPreference) {
        return TlCache.getOrElseUpdate(getCacheKey(id),
                () -> readPreference.map(rp -> collection.findById(id, rp)).getOrElse(() -> collection.findById(id))
                        .map(MongoGroup::to));
    }

    public ListF<Group> findAll(DjfsUid uid) {
        return find(uid, Option.of(ReadPreference.primary()));
    }

    public ListF<Group> find(DjfsUid uid, Option<ReadPreference> readPreference) {
        return TlCache.getOrElseUpdate(getCacheKey(uid),
                () -> readPreference.map(rp -> collection.find(Filters.eq("owner", uid.asString()), rp))
                        .getOrElse(() -> collection.find(Filters.eq("owner", uid.asString()))).map(MongoGroup::to));
    }

    @Override
    public void increaseSize(String id, long delta) {
        if (isReadonly.apply()) {
            throw new LegacyUserIsReadOnlyException(new DjfsException("Mongo Groups is in readonly"));
        }
        // todo: remove only necessary cache entry
        TlCache.flush();
        collection.updateOne(Filters.eq("_id", id), Updates.inc("size", delta));
    }

    public void insert(Group group) {
        if (isReadonly.apply()) {
            throw new LegacyUserIsReadOnlyException(new DjfsException("Mongo Groups is in readonly"));
        }
        TlCache.remove(getCacheKey(group.getOwner()));
        collection.insertOne(MongoGroup.cons(group));
    }

    public void remove(String id) {
        if (isReadonly.apply()) {
            throw new LegacyUserIsReadOnlyException(new DjfsException("Mongo Groups is in readonly"));
        }
        // todo: remove only necessary cache entry
        TlCache.flush();
        collection.deleteOne(id);
    }

    @Override
    public void deleteAll(DjfsUid uid) {
        if (isReadonly.apply()) {
            throw new LegacyUserIsReadOnlyException(new DjfsException("Mongo Groups is in readonly"));
        }
        // todo: remove only necessary cache entry
        TlCache.flush();
        collection.deleteMany(Filters.eq("owner", uid));
    }

    @Override
    public ListF<Group> findAll(ListF<String> groupIds) {
        ListF<Group> result = Cf.arrayList();

        SetF<String> notFoundGroups = groupIds.unique();
        for (String groupId : groupIds) {
            Group group = (Group) TlCache.get(groupId);
            if (group != null) {
                result.add(group);
                notFoundGroups.removeTs(groupId);
            }
        }

        if (!notFoundGroups.isEmpty()) {
            ListF<Group> groups = collection.findByIds(notFoundGroups).map(MongoGroup::to);
            groups.forEach(g -> TlCache.putSafe(getCacheKey(g.getId()), g));
            result.addAll(groups);
        }

        return result;
    }
}
