package ru.yandex.chemodan.app.djfs.core.share;

import com.mongodb.MongoClient;
import com.mongodb.ReadPreference;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.Updates;
import org.bson.BsonDocument;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.djfs.core.DjfsException;
import ru.yandex.chemodan.app.djfs.core.legacy.exception.LegacyUserIsReadOnlyException;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.commune.mongo3.MongoCollectionX;
import ru.yandex.misc.cache.tl.TlCache;

/**
 * @author eoshch
 */
public class MongoGroupLinkDao implements GroupLinkDao {
    private final MongoCollectionX<String, MongoGroupLink> collection;
    private final Function0<Boolean> isReadonly;

    public MongoGroupLinkDao(MongoClient mongoClient, Function0<Boolean> isReadonly) {
        this.collection = new MongoCollectionX<>(
                mongoClient.getDatabase("group_links").getCollection("group_links", BsonDocument.class),
                MongoGroupLink.B);
        this.isReadonly = isReadonly;
    }

    private String getCacheKey(DjfsUid uid) {
        return "mongo group link dao uid " + uid.asString();
    }

    private String getCacheKey(String groupId) {
        return "mongo group link dao groupId " + groupId;
    }

    @Override
    public Option<GroupLink> find(String id) {
        return collection.findById(id).map(MongoGroupLink::to);
    }

    @Override
    public ListF<GroupLink> findAll(DjfsUid uid) {
        return findAll(uid, Option.of(ReadPreference.primary()));
    }

    public ListF<GroupLink> findAll(DjfsUid uid, Option<ReadPreference> readPreference) {
        return TlCache.getOrElseUpdate(getCacheKey(uid),
                () -> readPreference.map(rp -> collection.find(Filters.eq("uid", uid.asString()), rp))
                        .getOrElse(() -> collection.find(Filters.eq("uid", uid.asString()))).map(MongoGroupLink::to));
    }

    @Override
    public ListF<GroupLink> findByGroupId(String groupId) {
        return findByGroupId(groupId, Option.of(ReadPreference.primary()));
    }

    public ListF<GroupLink> findByGroupId(String groupId, Option<ReadPreference> readPreference) {
        return TlCache.getOrElseUpdate(getCacheKey(groupId), () ->
                readPreference.map(rp -> collection.find(Filters.eq("gid", groupId), rp))
                        .getOrElse(() -> collection.find(Filters.eq("gid", groupId))).map(MongoGroupLink::to));
    }

    @Override
    public void insert(GroupLink groupLink) {
        if (isReadonly.apply()) {
            throw new LegacyUserIsReadOnlyException(new DjfsException("Mongo group_links is in readonly"));
        }
        TlCache.remove(getCacheKey(groupLink.getUid()));
        TlCache.remove(getCacheKey(groupLink.getGroupId()));
        collection.insertOne(MongoGroupLink.cons(groupLink));
    }

    @Override
    public void changePermissions(String groupLinkId, SharePermissions permissions) {
        if (isReadonly.apply()) {
            throw new LegacyUserIsReadOnlyException(new DjfsException("Mongo group_links is in readonly"));
        }
        TlCache.flush();
        collection.updateOneById(groupLinkId, Updates.set("rights", permissions.value()));
    }

    @Override
    public void deleteAll(DjfsUid uid) {
        if (isReadonly.apply()) {
            throw new LegacyUserIsReadOnlyException(new DjfsException("Mongo group_links is in readonly"));
        }
        TlCache.flush();
        collection.deleteMany(Filters.eq("uid", uid.asString()));
    }

    public void delete(String id) {
        if (isReadonly.apply()) {
            throw new LegacyUserIsReadOnlyException(new DjfsException("Mongo group_links is in readonly"));
        }
        TlCache.flush();
        collection.deleteOne(id);
    }
}
