package ru.yandex.chemodan.app.djfs.core.share;

import java.util.UUID;

import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDao;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.misc.cache.tl.TlCache;

/**
 * @author m-messiah
 */
public class PgGroupDao extends PgShardedDao implements GroupDao {
    private final static RowMapper<Group> M = (rs, rowNum) -> Group.builder()
            .id(UuidUtils.toHexString(UuidUtils.from(rs.getString("id"))))
            .owner(DjfsUid.cons(rs.getLong("owner")))
            .path(DjfsResourcePath.cons(rs.getLong("owner"), rs.getString("path")))
            .version(Option.ofNullable(rs.getLong("version")))
            .size(rs.getLong("size"))
            .build();

    public PgGroupDao(PgShardedDaoContext context) {
        super(context);
    }

    private String getCacheKey(DjfsUid uid) {
        return "pg group dao uid" + uid.asString();
    }

    private String getCacheKey(String groupId) {
        return "pg group dao groupId " + groupId;
    }


    @Override
    public void insert(Group group) {
        TlCache.remove(getCacheKey(group.getOwner()));
        String sql = collectStats(group)
                + " INSERT INTO disk.groups (id, owner, path, size, version) "
                + "VALUES (?, ?, ?, ?, ?)";

        jdbcTemplate(DjfsUid.COMMON_UID).update(sql,
                UuidUtils.fromHex(group.getId()), group.getOwner(), group.getPath().getPath(), group.getSize(), group.getVersion());
    }

    public void increaseSize(String id, long delta) {
        // todo: remove only necessary cache entry
        TlCache.flush();
        String sql = collectStats(DjfsUid.COMMON_UID) + " UPDATE disk.groups SET size = size + :delta WHERE id = :id";
        jdbcTemplate(DjfsUid.COMMON_UID).update(sql, Cf.map("id", UuidUtils.fromHex(id), "delta", delta));
    }

    public Option<Group> find(String id) {
        String sql = collectStats(DjfsUid.COMMON_UID) + " SELECT * FROM disk.groups WHERE id = :id";
        return TlCache.getOrElseUpdate(
                getCacheKey(id),
                () -> jdbcTemplate(DjfsUid.COMMON_UID).queryForOption(sql, M, Cf.map("id", UuidUtils.fromHex(id)))
        );
    }

    @Override
    public ListF<Group> findAll(ListF<String> groupIds) {
        ListF<Group> result = Cf.arrayList();

        SetF<String> notFoundGroups = groupIds.unique();
        for (String groupId : groupIds) {
            Group group = (Group) TlCache.get(groupId);
            if (group != null) {
                result.add(group);
                notFoundGroups.removeTs(groupId);
            }
        }

        if (!notFoundGroups.isEmpty()) {
            String sql = collectStats(DjfsUid.COMMON_UID) + " SELECT * FROM disk.groups WHERE id IN (:ids)";
            ListF<UUID> notFoundGroupsIds = notFoundGroups.map(UuidUtils::fromHex);
            ListF<Group> groups = jdbcTemplate(DjfsUid.COMMON_UID).query(sql, M, Cf.map("ids", notFoundGroupsIds));
            groups.forEach(g -> TlCache.putSafe(getCacheKey(g.getId()), g));
            result.addAll(groups);
        }
        return result;
    }

    @Override
    public ListF<Group> findAll(DjfsUid uid) {
        String sql = collectStats(uid) + " SELECT * FROM disk.groups WHERE owner = :uid";
        return TlCache.getOrElseUpdate(getCacheKey(uid), () -> jdbcTemplate(DjfsUid.COMMON_UID).query(sql, M, Cf.map("uid", uid)));
    }

    @Override
    public void deleteAll(DjfsUid uid) {
        String sql = collectStats(uid) + " DELETE FROM disk.groups WHERE owner = :uid";
        TlCache.flush();
        jdbcTemplate(DjfsUid.COMMON_UID).update(sql, Cf.map("uid", uid));
    }

    @Override
    public void remove(String id) {
        String sql = collectStats(DjfsUid.COMMON_UID) + " DELETE FROM disk.groups WHERE id = :id";
        TlCache.flush();
        jdbcTemplate(DjfsUid.COMMON_UID).update(sql, Cf.map("id", UuidUtils.fromHex(id)));
    }
}
