package ru.yandex.chemodan.app.djfs.core.share;

import org.joda.time.Instant;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDao;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.misc.cache.tl.TlCache;

public class PgGroupLinkDao extends PgShardedDao implements GroupLinkDao {
    private final static RowMapper<GroupLink> M = (rs, rowNum) -> GroupLink.builder()
            .id(UuidUtils.toHexString(UuidUtils.from(rs.getString("id"))))
            .groupId(UuidUtils.toHexString(UuidUtils.from(rs.getString("gid"))))
            .uid(DjfsUid.cons(rs.getLong("uid")))
            .path(DjfsResourcePath.cons(rs.getLong("uid"), rs.getString("path")))
            .version(rs.getLong("version"))
            .permissions(SharePermissions.R.fromValue(rs.getInt("rights")))
            .creationTime(new Instant(rs.getTimestamp("date_created")))
            .b2bKey(Option.ofNullable(rs.getString("b2b_key")))
            .universeLogin(Option.ofNullable(rs.getString("universe_login")))
            .universeService(Option.ofNullable(rs.getString("universe_service")))
            .build();

    public PgGroupLinkDao(PgShardedDaoContext context) {
        super(context);
    }

    private String getCacheKey(DjfsUid uid) {
        return "pg group link dao uid " + uid.asString();
    }

    private String getCacheKey(String groupId) {
        return "pg group link dao groupId " + groupId;
    }


    @Override
    public void insert(GroupLink groupLink) {
        TlCache.remove(getCacheKey(groupLink.getUid()));
        TlCache.remove(getCacheKey(groupLink.getGroupId()));

        String sql = collectStats(groupLink)
                + " INSERT INTO disk.group_links (id, gid, uid, path, version, rights, b2b_key, universe_login, universe_service, date_created) "
                + "VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

        jdbcTemplate(DjfsUid.COMMON_UID).update(sql,
                UuidUtils.fromHex(groupLink.getId()),
                UuidUtils.fromHex(groupLink.getGroupId()),
                groupLink.getUid(),
                groupLink.getPath().getPath(),
                groupLink.getVersion(),
                groupLink.getPermissions().value(),
                groupLink.getB2bKey(),
                groupLink.getUniverseLogin(),
                groupLink.getUniverseService(),
                groupLink.getCreationTime()
        );
    }

    @Override
    public ListF<GroupLink> findAll(DjfsUid uid) {
        String sql = collectStats(uid) + " SELECT * FROM disk.group_links WHERE uid = :uid";
        return TlCache.getOrElseUpdate(getCacheKey(uid), () -> jdbcTemplate(DjfsUid.COMMON_UID).query(sql, M, Cf.map("uid", uid)));
    }

    @Override
    public Option<GroupLink> find(String id) {
        String sql = collectStats(DjfsUid.COMMON_UID) + " SELECT * FROM disk.group_links WHERE id = :id";
        return TlCache.getOrElseUpdate(
                getCacheKey(id),
                () -> jdbcTemplate(DjfsUid.COMMON_UID).queryForOption(sql, M, Cf.map("id", UuidUtils.fromHex(id)))
        );
    }

    @Override
    public ListF<GroupLink> findByGroupId(String groupId) {
        String sql = collectStats(DjfsUid.COMMON_UID) + " SELECT * FROM disk.group_links WHERE gid = :gid";
        return TlCache.getOrElseUpdate(
                getCacheKey(groupId),
                () -> jdbcTemplate(DjfsUid.COMMON_UID).queryForOption(sql, M, Cf.map("gid", UuidUtils.fromHex(groupId)))
        );
    }

    @Override
    public void changePermissions(String id, SharePermissions permissions) {
        TlCache.flush();
        String sql = collectStats(DjfsUid.COMMON_UID) + " UPDATE disk.group_links SET rights = :rights WHERE id = :id";
        jdbcTemplate(DjfsUid.COMMON_UID).update(sql, Cf.map("id", UuidUtils.fromHex(id), "rights", permissions.value()));
    }

    @Override
    public void deleteAll(DjfsUid uid) {
        String sql = collectStats(uid) + " DELETE FROM disk.group_links WHERE uid = :uid";
        TlCache.flush();
        jdbcTemplate(DjfsUid.COMMON_UID).update(sql, Cf.map("uid", uid));
    }

    @Override
    public void delete(String id) {
        String sql = collectStats(DjfsUid.COMMON_UID) + " DELETE FROM disk.group_links WHERE id = :id";
        TlCache.flush();
        jdbcTemplate(DjfsUid.COMMON_UID).update(sql, Cf.map("id", UuidUtils.fromHex(id)));
    }
}
