package ru.yandex.chemodan.app.djfs.core.share;

import java.util.Objects;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.impl.ArrayListF;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;

/**
 * @author eoshch
 */
public class ShareInfo {
    private Group group;
    private MapF<DjfsUid, GroupLink> groupLinks;

    public ShareInfo(Group group, ListF<GroupLink> groupLinks) {
        this.group = group;
        this.groupLinks = groupLinks.toMapMappingToKey(GroupLink::getUid);
    }

    public String getGroupId() {
        return group.getId();
    }

    public Option<String> getGroupLinkId(DjfsUid uid) {
        return groupLinks.getO(uid).map(GroupLink::getId);
    }

    public Option<Long> getGroupLinkBaseVersion(DjfsUid uid) {
        return groupLinks.getO(uid).filterMap(GroupLink::getSearchIndexerBaseVersion);
    }

    public DjfsResourcePath getGroupPath() {
        return group.getPath();
    }

    public Long getGroupSize() {
        return group.getSize();
    }

    public ListF<DjfsUid> allUids() {
        ListF<DjfsUid> result = new ArrayListF<>();
        result.add(group.getUid());
        result.addAll(groupLinks.keys());
        return result;
    }

    public DjfsUid getOwnerUid() {
        return group.getUid();
    }

    public Option<DjfsResourcePath> participantPathToOwnerPath(DjfsResourcePath participantPath) {
        Option<GroupLink> groupLinkO = groupLinks.getO(participantPath.getUid());
        if (groupLinkO.isPresent()) {
            GroupLink groupLink = groupLinkO.get();
            if (groupLink.getPath().equals(participantPath)) {
                return Option.of(group.getPath());
            } else if (groupLink.getPath().isParentFor(participantPath)) {
                return Option.of(participantPath.changeParent(groupLink.getPath(), group.getPath()));
            }
        }
        return Option.empty();
    }

    public Option<DjfsResourcePath> getRootPath(DjfsUid uid) {
        if (Objects.equals(uid, group.getUid())) {
            return Option.of(group.getPath());
        }
        return groupLinks.getO(uid).map(GroupLink::getPath);
    }

    public boolean isOwnerRoot(DjfsResourcePath path) {
        return getGroupPath().equals(path);
    }

    public boolean isParticipantRoot(DjfsResourcePath participantPath) {
        Option<DjfsResourcePath> participantRootPathO = getRootPath(participantPath.getUid());
        if (participantRootPathO.isPresent()) {
            return participantRootPathO.get().equals(participantPath);
        }
        return false;
    }

    public boolean isRoot(DjfsResourcePath path) {
        return isOwnerRoot(path) || isParticipantRoot(path);
    }

    public Option<DjfsResourcePath> ownerPathToParticipantPath(DjfsResourcePath ownerPath, DjfsUid participantUid) {
        // todo: check that path is indeed owner's?
        if (Objects.equals(ownerPath.getUid(), participantUid)) {
            return Option.of(ownerPath);
        }
        Option<GroupLink> groupLinkO = groupLinks.getO(participantUid);
        if (ownerPath.equals(group.getPath())) {
            return groupLinkO.map(GroupLink::getPath);
        }
        return groupLinkO.map(x -> ownerPath.changeParent(group.getPath(), x.getPath()));
    }

    public Option<SharePermissions> getPermissions(DjfsUid uid) {
        if (Objects.equals(group.getUid(), uid)) {
            return Option.of(SharePermissions.READ_WRITE);
        }
        Option<GroupLink> groupLinkO = groupLinks.getO(uid);
        return groupLinkO.map(GroupLink::getPermissions);
    }

    public Option<SharePermissions> getGroupLinkPermissionsForPath(DjfsUid uid, DjfsResourcePath path) {
        return groupLinks.getO(uid).filter(groupLink -> path.isParentFor(groupLink.getPath())).map(GroupLink::getPermissions);
    }

    public boolean suitesUid(Option<DjfsUid> participantUid) {
        if (!participantUid.isPresent()) {
            return true;
        }
        return getGroupLinkId(participantUid.get()).isPresent();
    }
}
