package ru.yandex.chemodan.app.djfs.core.share;

import com.mongodb.ReadPreference;
import lombok.RequiredArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;

/**
 * @author eoshch
 */
@RequiredArgsConstructor
public class ShareInfoManager {
    private final GroupDao groupDao;
    private final GroupLinkDao groupLinkDao;

    private ShareInfo shareInfo(Group group) {
        ListF<GroupLink> groupLinks = groupLinkDao.findByGroupId(group.getId());
        return new ShareInfo(group, groupLinks);
    }

    private Option<ShareInfo> shareInfo(GroupLink groupLink) {
        return groupDao.find(groupLink.getGroupId()).map(this::shareInfo);
    }

    @Deprecated
    public Option<ShareInfo> get(String groupId) {
        return groupDao.find(groupId).map(this::shareInfo);
    }

    @Deprecated
    public Option<ShareInfo> get(DjfsResource resource) {
        return get(resource.getPath());
    }

    @Deprecated
    public Option<ShareInfo> get(DjfsResource resource, Option<ReadPreference> readPreference) {
        return get(resource.getPath(), readPreference);
    }

    @Deprecated
    public Option<ShareInfo> get(DjfsResourcePath path) {
        return get(path, Option.of(ReadPreference.primary()));
    }

    @Deprecated
    public Option<ShareInfo> get(DjfsResourcePath path, Option<ReadPreference> readPreference) {
        return get(path, groupDao.findAll(path.getUid()),
                groupLinkDao.findAll(path.getUid()));
    }

    @Deprecated
    public Option<ShareInfo> get(DjfsResourcePath path, ListF<Group> uidGroups, ListF<GroupLink> uidGroupLinks)
    {
        // todo: optimize: shares exist only for /disk for now
        for (GroupLink groupLink : uidGroupLinks) {
            if (groupLink.getPath().isParentFor(path)) {
                Group group = groupDao.find(groupLink.getGroupId()).get();
                ListF<GroupLink> links = groupLinkDao.findByGroupId(group.getId());
                return Option.of(new ShareInfo(group, links));
            }
        }

        for (Group group : uidGroups) {
            if (group.getPath().isParentFor(path)) {
                ListF<GroupLink> links = groupLinkDao.findByGroupId(group.getId());
                return Option.of(new ShareInfo(group, links));
            }
        }
        return Option.empty();
    }

    public Option<ShareInfo> getWithRoot(DjfsResource resource, Option<DjfsUid> participantUid) {
        return getWithRoot(resource.getPath(), participantUid);
    }

    public Option<ShareInfo> getWithRoot(DjfsResourcePath path, Option<DjfsUid> participantUid) {
        return getWithRoot(path, groupDao.findAll(path.getUid()),
                groupLinkDao.findAll(path.getUid()), participantUid);
    }

    public Option<ShareInfo> getWithRoot(DjfsResourcePath path, ListF<Group> uidGroups, ListF<GroupLink> uidGroupLinks,
                                         Option<DjfsUid> participantUid)
    {
        // the shareInfo.suitesUid call is a hack for CHEMODAN-73101 purposes

        for (GroupLink groupLink : uidGroupLinks) {
            if (groupLink.getPath().isParentFor(path) || groupLink.getPath().equals(path)) {
                final Option<Group> groupO = groupDao.find(groupLink.getGroupId());
                if (groupO.isPresent()) {
                    ListF<GroupLink> links = groupLinkDao.findByGroupId(groupO.get().getId());
                    ShareInfo shareInfo = new ShareInfo(groupO.get(), links);
                    if (shareInfo.suitesUid(participantUid)) {
                        return Option.of(shareInfo);
                    }
                }
            }
        }

        for (Group group : uidGroups) {
            if (group.getPath().isParentFor(path) || group.getPath().equals(path)) {
                ListF<GroupLink> links = groupLinkDao.findByGroupId(group.getId());
                ShareInfo shareInfo = new ShareInfo(group, links);
                if (shareInfo.suitesUid(participantUid)) {
                    return Option.of(new ShareInfo(group, links));
                }
            }
        }
        return Option.empty();
    }

    public Option<ShareInfo> getWithoutParent(DjfsResourcePath path) {
        for (GroupLink groupLink : groupLinkDao.findAll(path.getUid())) {
            if (groupLink.getPath().equals(path)) {
                final Option<Group> groupO = groupDao.find(groupLink.getGroupId());
                if (groupO.isPresent()) {
                    ListF<GroupLink> links = groupLinkDao.findByGroupId(groupO.get().getId());
                    return Option.of(new ShareInfo(groupO.get(), links));
                }
            }
        }

        for (Group group : groupDao.findAll(path.getUid())) {
            if (group.getPath().equals(path)) {
                ListF<GroupLink> links = groupLinkDao.findByGroupId(group.getId());
                return Option.of(new ShareInfo(group, links));
            }
        }
        return Option.empty();
    }

    public ListF<ShareInfo> get(DjfsUid uid) {
        // todo: optimize: bulk reads
        ListF<Group> groups = groupDao.findAll(uid);
        ListF<GroupLink> groupLinks = groupLinkDao.findAll(uid);
        return groups.map(this::shareInfo).plus(groupLinks.filterMap(this::shareInfo));
    }

    public ListF<Group> getJoinedGroups(DjfsUid uid) {
        ListF<GroupLink> groupLinks = groupLinkDao.findAll(uid);
        return groupDao.findAll(groupLinks.map(GroupLink::getGroupId));
    }

    public ListF<ShareInfo> getParticipantSharedSubfolders(DjfsResourcePath path) {
        ListF<ShareInfo> list = Cf.arrayList();
        ListF<GroupLink> groupLinks = groupLinkDao.findAll(path.getUid());
        for (GroupLink groupLink : groupLinks) {
            if (path.isParentFor(groupLink.getPath()) || path.equals(groupLink.getPath())) {
                Option<ShareInfo> shareInfo = shareInfo(groupLink);
                if (shareInfo.isPresent()) {
                    list.add(shareInfo.get());
                }
            }
        }
        return list;
    }
}
