package ru.yandex.chemodan.app.djfs.core.share;

import java.util.Objects;

import lombok.RequiredArgsConstructor;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.EventManager;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsPrincipal;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.FilesystemAccessController;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderDjfsResource;
import ru.yandex.chemodan.app.djfs.core.share.event.UserKickedFromGroupEvent;
import ru.yandex.chemodan.app.djfs.core.share.event.UserLeftGroupEvent;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.user.UserDao;
import ru.yandex.chemodan.app.djfs.core.util.InstantUtils;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author eoshch
 */
@RequiredArgsConstructor
public class ShareManager {
    private static final Logger logger = LoggerFactory.getLogger(ShareManager.class);

    private final GroupDao groupDao;
    private final GroupLinkDao groupLinkDao;
    private final DjfsResourceDao djfsResourceDao;
    private final UserDao userDao;

    private final ShareInfoManager shareInfoManager;
    private final FilesystemAccessController filesystemAccessController;
    private final EventManager eventManager;

    /**
     * Use in tests only!
     * Not ready for production!
     */
    public Group createGroup(DjfsResourcePath path) {
        Group group = Group.builder()
                .id(UuidUtils.randomToHexString())
                .owner(path.getUid())
                .path(path)
                .version(Option.empty())
                .size(0)
                .build();
        groupDao.insert(group);
        return group;
    }

    /**
     * Use in tests only!
     * Not ready for production!
     */
    public GroupLink createLink(String groupId, DjfsResourcePath path, SharePermissions permissions) {
        GroupLink groupLink = GroupLink.builder()
                .id(UuidUtils.randomToHexString())
                .groupId(groupId)
                .uid(path.getUid())
                .path(path)
                .version(1)
                .permissions(permissions)
                .creationTime(Instant.now())
                .build();
        groupLinkDao.insert(groupLink);
        return groupLink;
    }

    /**
     * Use in tests only!
     * Not ready for production!
     */
    public void changeLinkPermissions(String groupLinkId, SharePermissions permissions) {
        groupLinkDao.changePermissions(groupLinkId, permissions);
    }

    public void leave(DjfsPrincipal principal, String groupId, DjfsUid uid) {
        Option<ShareInfo> shareInfo = shareInfoManager.get(groupId);
        if (!shareInfo.isPresent()) {
            return;
        }
        leave(principal, shareInfo.get(), uid);
    }

    public void leave(DjfsPrincipal principal, ShareInfo shareInfo, DjfsUid uid) {
        filesystemAccessController.share.checkLeave(principal, uid, shareInfo);

        Option<String> groupLinkId = shareInfo.getGroupLinkId(uid);
        if (!groupLinkId.isPresent()) {
            return;
        }

        logger.info("leave as " + uid.asString() + " from group " + shareInfo.getGroupId());

        Instant now = Instant.now();
        UserLeftGroupEvent.UserLeftGroupEventBuilder event = UserLeftGroupEvent.builder()
                .uid(uid)
                .shareInfo(shareInfo)
                .instant(now);

        groupLinkDao.delete(groupLinkId.get());
        // todo: should we move folder deletion to filesystem? event handling is different
        DjfsResourcePath rootPath = shareInfo.getRootPath(uid).get();
        Option<DjfsResource> root = djfsResourceDao.find(rootPath);
        if (root.isPresent() && (root.get() instanceof FolderDjfsResource)) {
            djfsResourceDao.delete(rootPath);
            event.userShareRoot(root.cast());
        }
        userDao.incrementVersionTo(uid, InstantUtils.toVersion(now));

        eventManager.send(event.build());
    }

    public void kick(DjfsPrincipal principal, String groupId, DjfsUid uid) {
        Option<ShareInfo> shareInfo = shareInfoManager.get(groupId);
        if (!shareInfo.isPresent()) {
            return;
        }
        kick(principal, shareInfo.get(), uid);
    }

    public void kick(DjfsPrincipal principal, ShareInfo shareInfo, DjfsUid uid) {
        filesystemAccessController.share.checkKick(principal, uid, shareInfo);

        Option<String> groupLinkId = shareInfo.getGroupLinkId(uid);
        if (!groupLinkId.isPresent()) {
            return;
        }

        logger.info("kick " + uid.asString() + " from group " + shareInfo.getGroupId());

        Instant now = Instant.now();
        UserKickedFromGroupEvent.UserKickedFromGroupEventBuilder event = UserKickedFromGroupEvent.builder()
                .uid(uid)
                .shareInfo(shareInfo)
                .instant(now);

        groupLinkDao.delete(groupLinkId.get());
        // todo: should we move folder deletion to filesystem? event handling is different
        DjfsResourcePath rootPath = shareInfo.getRootPath(uid).get();
        Option<DjfsResource> root = djfsResourceDao.find(rootPath);
        if (root.isPresent() && (root.get() instanceof FolderDjfsResource)) {
            djfsResourceDao.delete(rootPath);
            event.userShareRoot(root.cast());
        }
        userDao.incrementVersionTo(uid, InstantUtils.toVersion(now));

        eventManager.send(event.build());
    }

    public void removeAllSharedFolders(DjfsPrincipal principal, DjfsUid uid) {
        filesystemAccessController.share.checkRemoveAllSharedFolders(principal, uid);

        logger.info("remove all shared folders of " + uid.asString());

        ListF<ShareInfo> shareInfos = shareInfoManager.get(uid);
        for (ShareInfo shareInfo : shareInfos) {
            if (Objects.equals(uid, shareInfo.getOwnerUid())) {
                for (DjfsUid participantUid : shareInfo.allUids().filterNot(x -> Objects.equals(x, uid))) {
                    kick(DjfsPrincipal.SYSTEM, shareInfo, participantUid);
                }
                groupDao.remove(shareInfo.getGroupId());
            } else {
                leave(DjfsPrincipal.SYSTEM, shareInfo, uid);
            }
        }
    }
}
