package ru.yandex.chemodan.app.djfs.core.tasks;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.djfs.core.operations.MpfsOperationHandler;
import ru.yandex.chemodan.app.djfs.core.operations.Operation;
import ru.yandex.chemodan.app.djfs.core.operations.OperationDao;
import ru.yandex.chemodan.app.djfs.core.operations.OperationNotFoundException;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.bazinga.YcridOnetimeTaskSupport;
import ru.yandex.chemodan.bazinga.YcridTaskParameters;
import ru.yandex.commune.bazinga.BazingaStopExecuteException;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author yashunsky
 */
public abstract class DjfsOperationTaskSupport<T extends YcridTaskParameters & DjfsOperationTaskParameters> extends YcridOnetimeTaskSupport<T> {
    private static final Logger logger = LoggerFactory.getLogger(DjfsOperationTaskSupport.class);

    private final OperationDao operationDao;

    private static final SetF<Operation.State> ACCEPT_STATES =
            Cf.set(Operation.State.WAITING, Operation.State.EXECUTING);

    public DjfsOperationTaskSupport(T parameters) {
        super(parameters);
        this.operationDao = null;
    }

    protected DjfsOperationTaskSupport(Class<T> parametersClass, OperationDao operationDao) {
        super(parametersClass);
        this.operationDao = operationDao;
    }

    @Override
    protected void doExecute(T parameters, ExecutionContext context) {
        DjfsUid uid = parameters.getUid();
        String oid = parameters.getOid();

        Operation operation = operationDao.find(uid, oid).getOrThrow(() -> new OperationNotFoundException(uid, oid));

        if (!ACCEPT_STATES.containsTs(operation.getState())) {
            logger.info("Operation {} for uid {} is in state {}, skipping", oid, uid, operation.getState());
            return;
        }

        operationDao.changeState(uid, oid, Operation.State.WAITING, Operation.State.EXECUTING);

        MpfsOperationHandler.Status status;
        try {
            status = executeInner(parameters, context);
        } catch (Throwable t) {
            try {
                operationDao.changeState(uid, oid, Operation.State.EXECUTING, Operation.State.FAILED);
                throw new BazingaStopExecuteException(t);
            } catch (Throwable tr) {
                tr.addSuppressed(t);
                throw tr;
            }
        }
        Operation.State state = status == MpfsOperationHandler.Status.DONE
                ? Operation.State.COMPLETED
                : Operation.State.FAILED;
        operationDao.changeState(uid, oid, Operation.State.EXECUTING, state);
    }

    abstract protected MpfsOperationHandler.Status executeInner(T parameters, ExecutionContext context);
}
