package ru.yandex.chemodan.app.djfs.core.user;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.ActionContext;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.annotation.BenderTextValue;
import ru.yandex.misc.lang.Assume;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author eoshch
 */
public class DjfsUid extends DefaultObject {
    public static DjfsUid SHARE_PRODUCTION = new DjfsUid("share_production", 1L, true);
    public static DjfsUid COMMON_UID = new DjfsUid("-1", -1L, false);

    private static final ListF<DjfsUid> SPECIAL_UIDS = Cf.arrayList(
            SHARE_PRODUCTION,
            COMMON_UID
    );

    private static final MapF<String, DjfsUid> SPECIAL_UID_STRING_MAP = SPECIAL_UIDS
            .toMapMappingToKey(DjfsUid::asString);
    private static final MapF<Long, DjfsUid> SPECIAL_UID_LONG_MAP = SPECIAL_UIDS.toMapMappingToKey(DjfsUid::asLong);

    private static final Assume INTERNAL_ASSUME = new Assume() {
        @Override
        public void fail(String message) {
            throw new InvalidDjfsUidException(message);
        }
    };

    private static final Assume CLIENT_ASSUME = new Assume() {
        @Override
        public void fail(String message) {
            throw new InvalidClientInputDjfsUidException(message);
        }
    };

    private final long longUid;
    private final String stringUid;
    private final boolean shared;

    private DjfsUid(String stringUid, long longUid, boolean shared) {
        this.stringUid = stringUid;
        this.longUid = longUid;
        this.shared = shared;
    }

    public long asLong() {
        return this.longUid;
    }

    public String asString() {
        return this.stringUid;
    }

    public PassportUid asPassportUid() {
        return PassportUid.cons(this.longUid);
    }

    public MpfsUser asMpfsUser() {
        return MpfsUser.of(stringUid);
    }

    @BenderTextValue
    @Override
    public String toString() {
        return this.stringUid;
    }

    public static DjfsUid cons(long uid) {
        return cons(uid, INTERNAL_ASSUME);
    }

    public static DjfsUid consShared(long uid, String stringUid) {
        return new DjfsUid(stringUid, uid, true);
    }

    public static DjfsUid cons(long uid, Assume assume) {
        if (SPECIAL_UID_LONG_MAP.containsKeyTs(uid)) {
            return SPECIAL_UID_LONG_MAP.getTs(uid);
        }

        assume.isTrue(uid > 0, String.valueOf(uid));
        return new DjfsUid(String.valueOf(uid), uid, false);
    }

    @BenderTextValue
    public static DjfsUid cons(String uid) {
        return cons(uid, INTERNAL_ASSUME);
    }

    public static DjfsUid cons(String uid, ActionContext context) {
        switch (context) {
            case INTERNAL:
                return cons(uid, INTERNAL_ASSUME);
            case CLIENT_INPUT:
                return cons(uid, CLIENT_ASSUME);
        }
        throw new NotImplementedException();
    }

    public static DjfsUid cons(String uid, Assume assume) {
        if (SPECIAL_UID_STRING_MAP.containsKeyTs(uid)) {
            return SPECIAL_UID_STRING_MAP.getTs(uid);
        }

        assume.notBlank(uid, uid);
        assume.isTrue(StringUtils.isNumericArabic(uid), uid);

        try {
            long longUid = Long.parseLong(uid);

            if (SPECIAL_UID_LONG_MAP.containsKeyTs(longUid)) {
                return SPECIAL_UID_LONG_MAP.getTs(longUid);
            }

            return new DjfsUid(uid, longUid, false);
        } catch (NumberFormatException e) {
            assume.fail(uid + ": " + e.getMessage());
        }
        // this is unreachable
        throw new NotImplementedException();
    }

    public static boolean isValid(String uid) {
        return SPECIAL_UID_STRING_MAP.containsKeyTs(uid)
                || (!StringUtils.isBlank(uid) && StringUtils.isNumericArabic(uid));
    }

    public static boolean isValid(long uid) {
        return SPECIAL_UID_LONG_MAP.containsKeyTs(uid) || uid > 0;
    }

    public boolean isShared() {
        return this.shared;
    }
}
