package ru.yandex.chemodan.app.djfs.core.user;

import com.mongodb.client.model.Filters;
import com.mongodb.client.model.Updates;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.db.mongo.MongoShardedDao;
import ru.yandex.chemodan.app.djfs.core.db.mongo.MongoShardedDaoContext;

/**
 * @author eoshch
 */
public class MongoUserDao extends MongoShardedDao<String, MongoUser> implements UserDao {

    public MongoUserDao(MongoShardedDaoContext dependencies) {
        super(dependencies, "user_index", "user_index", MongoUser.B);
    }

    @Override
    public Option<UserData> find(DjfsUid uid) {
        return collectionX(uid).findById(uid.asString()).map(MongoUser::toUserData);
    }

    @Override
    public void insert(UserData user) {
        collectionX(user.getId()).insertOne(MongoUser.cons(user));
    }

    @Override
    public void changeType(DjfsUid uid, UserType type) {
        collectionX(uid).updateOneById(uid.asString(), Updates.set("type", type.value()));
    }

    @Override
    public void incrementVersionTo(DjfsUid uid, long version) {
        collectionX(uid).updateOne(
                Filters.and(
                        Filters.eq("_id", uid.asString()),
                        Filters.or(Filters.exists("version", false), Filters.lt("version", version))
                ), Updates.set("version", version));
    }

    @Override
    public Option<Long> incrementVersionTo_ReturnOld(DjfsUid uid, long version) {
        return collectionX(uid).findOneAndUpdate(
                Filters.and(
                        Filters.eq("_id", uid.asString()),
                        Filters.or(Filters.exists("version", false), Filters.lt("version", version))
                ), Updates.set("version", version), false, false).map(MongoUser::toUserData).filterMap(x -> x.getVersion());
    }

    @Override
    public void setVersion(DjfsUid uid, long version) {
        collectionX(uid).updateOne(Filters.eq("_id", uid.asString()), Updates.set("version", version));
    }

    @Override
    public void setMinimumDeltaVersion(DjfsUid uid, long version) {
        collectionX(uid).updateOne(Filters.eq("_id", uid.asString()), Updates.set("last_quick_move_version", version));
    }

    @Override
    public void setB2bKey(DjfsUid uid, String b2bKey) {
        collectionX(uid).updateOne(Filters.eq("_id", uid.asString()), Updates.set("b2b_key", b2bKey));
    }

    @Override
    public void setDeleted(DjfsUid uid) {
        collectionX(uid).updateOne(Filters.and(Filters.eq("_id", uid.asString()), Filters.exists("deleted", false)),
                Updates.set("deleted", Instant.now().toDate()));
    }

    @Override
    public void block(DjfsUid uid) {
        collectionX(uid).updateOne(Filters.and(Filters.eq("_id", uid.asString())), Updates.set("blocked", 1));
    }

    @Override
    public void unblock(DjfsUid uid) {
        collectionX(uid).updateOne(Filters.and(Filters.eq("_id", uid.asString())), Updates.unset("blocked"));
    }

    @Override
    public void addCollection(DjfsUid uid, String collection) {
        collectionX(uid).updateOneById(uid.asString(), Updates.addToSet("collections", collection));
    }

    @Override
    public void setQuickMoveFlag(DjfsUid uid) {
        throw new NotImplementedException();
    }

    @Override
    public Option<Long> getFaceClustersVersion(DjfsUid uid) {
        throw new NotImplementedException();
    }

    @Override
    public void updateFaceClustersVersion(DjfsUid uid, long version) {
        throw new NotImplementedException();
    }

    @Override
    public void removeFaceClustersVersion(DjfsUid uid) {
        throw new NotImplementedException();
    }

    @Override
    public Tuple2<FacesIndexingState, Instant> getFacesIndexingState(DjfsUid uid) {
        throw new NotImplementedException();
    }

    @Override
    public void setFacesIndexingState(DjfsUid uid, FacesIndexingState state) {
        throw new NotImplementedException();
    }
}
