package ru.yandex.chemodan.app.djfs.core.user;

import java.util.Map;

import org.joda.time.Instant;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDao;
import ru.yandex.chemodan.app.djfs.core.db.pg.PgShardedDaoContext;
import ru.yandex.chemodan.app.djfs.core.db.pg.ResultSetUtils;

/**
 * @author eoshch
 */
public class PgUserDao extends PgShardedDao implements UserDao {

    private final static RowMapper<UserData> M = (rs, rowNum) ->
        // todo: handle locales
        new UserData(DjfsUid.cons(rs.getString("uid")), ResultSetUtils.getInstantO(rs, "reg_time"),
                Option.ofNullable(rs.getString("locale")).map(UserLocale.R_BY_VALUE::valueOf),
                rs.getInt("shard_key"), Cf.wrap((String[]) rs.getArray("collections").getArray()), Option.ofNullable(rs.getString("user_type")).map(UserType.R::fromValue),
                ResultSetUtils.getLongO(rs, "version"), ResultSetUtils.getLongO(rs, "last_quick_move_version"), rs.getBoolean("blocked"),
                ResultSetUtils.getInstantO(rs, "deleted"), ResultSetUtils.getStringO(rs, "b2b_key"),
                ResultSetUtils.getBooleanO(rs, "is_reindexed_for_quick_move").getOrElse(false), true,
                ResultSetUtils.getStringO(rs, "yateam_uid"), Option.ofNullable(rs.getString("pdd")).map(Pdd::fromJsonString),
                rs.getBoolean("is_paid"));

    public PgUserDao(PgShardedDaoContext dependencies) {
        super(dependencies);
    }

    @Override
    public Option<UserData> find(DjfsUid uid) {
        String sql = collectStats(uid) + " SELECT * FROM disk.user_index WHERE uid = :uid";
        Map<String, Object> parameters = Cf.map("uid", uid);
        return jdbcTemplate(uid).queryForOption(sql, M, parameters);
    }

    @Override
    public void insert(UserData user) {
        String sql = collectStats(user)
                + " INSERT INTO disk.user_index (uid, version, blocked, deleted, user_type, reg_time, locale, "
                + " shard_key, b2b_key, pdd, yateam_uid, collections, last_quick_move_version) "
                + " VALUES (:uid, :version, :blocked, :deleted, :user_type::disk.user_types, :reg_time, "
                + " :locale::disk.locales, :shard_key, :b2b_key, :pdd::json, :yateam_uid, '{}', :last_quick_move_version)";

        // todo: set collections array
        Map<String, Object> parameters = Cf.toMap(Tuple2List.fromPairs(
                "uid", user.getUid(),
                "version", user.getVersion().getOrNull(),
                "blocked", user.isBlocked(),
                "deleted", user.getDeleted().getOrNull(),
                "user_type", user.getType().getOrNull(),
                "reg_time", user.getRegTime().getOrNull(),
                "locale", user.getLocale().getOrNull(),
                "shard_key", user.getShardKey(),
                "b2b_key", user.getB2bKey().getOrNull(),
                "pdd", user.getPdd().map(Pdd::toPgJsonString).getOrNull(),
                "yateam_uid", user.getYateamUid().getOrNull(),
                "last_quick_move_version", user.getMinimumDeltaVersion().getOrNull()
        ));

        jdbcTemplate(user.getUid()).update(sql, parameters);
    }

    @Override
    public void changeType(DjfsUid uid, UserType type) {
        String sql = collectStats(uid)
                + " UPDATE disk.user_index SET user_type = :user_type::disk.user_types WHERE uid = :uid";
        Map<String, Object> parameters = Cf.map("uid", uid, "user_type", type);
        jdbcTemplate(uid).update(sql, parameters);
    }

    @Override
    public void incrementVersionTo(DjfsUid uid, long version) {
        String sql = collectStats(uid) + " UPDATE disk.user_index SET version = :version "
                + " WHERE uid = :uid AND (version < :version OR version IS NULL)";
        Map<String, Object> parameters = Cf.map("uid", uid, "version", version);
        jdbcTemplate(uid).update(sql, parameters);
    }

    @Override
    public Option<Long> incrementVersionTo_ReturnOld(DjfsUid uid, long version) {
        // https://stackoverflow.com/questions/7923237/return-pre-update-column-values-using-sql-only-postgresql-version
        String sql = collectStats(uid) + " UPDATE disk.user_index x SET version = :version "
                + " FROM ("
                + "     SELECT uid, version"
                + "     FROM disk.user_index"
                + "     WHERE uid = :uid AND (version < :version OR version IS NULL) FOR UPDATE) y"
                + " WHERE x.uid = y.uid"
                + " RETURNING y.version";
        Map<String, Object> parameters = Cf.map("uid", uid, "version", version);
        ListF<Option<Long>> result = jdbcTemplate(uid).query(sql, (x, y) -> ResultSetUtils.getLongO(x, "version"), parameters);
        return result.singleO().filterMap(x -> x);
    }

    @Override
    public void setVersion(DjfsUid uid, long version) {
        String sql = collectStats(uid) + " UPDATE disk.user_index SET version = :version WHERE uid = :uid";
        Map<String, Object> parameters = Cf.map("uid", uid, "version", version);
        jdbcTemplate(uid).update(sql, parameters);
    }

    @Override
    public void setMinimumDeltaVersion(DjfsUid uid, long version) {
        String sql = collectStats(uid)
                + " UPDATE disk.user_index SET last_quick_move_version = :last_quick_move_version WHERE uid = :uid";
        Map<String, Object> parameters = Cf.map("uid", uid, "last_quick_move_version", version);
        jdbcTemplate(uid).update(sql, parameters);
    }

    @Override
    public void setB2bKey(DjfsUid uid, String b2bKey) {
        String sql = collectStats(uid) + " UPDATE disk.user_index SET b2b_key = :b2b_key WHERE uid = :uid";
        Map<String, Object> parameters = Cf.map("uid", uid, "b2b_key", b2bKey);
        jdbcTemplate(uid).update(sql, parameters);
    }

    @Override
    public void setDeleted(DjfsUid uid) {
        String sql = collectStats(uid) + " UPDATE disk.user_index SET deleted = :deleted WHERE uid = :uid and deleted is null";
        Map<String, Object> parameters = Cf.map("uid", uid, "deleted", Instant.now());
        jdbcTemplate(uid).update(sql, parameters);
    }

    @Override
    public void block(DjfsUid uid) {
        String sql = collectStats(uid) + " UPDATE disk.user_index SET blocked = true WHERE uid = :uid";
        Map<String, Object> parameters = Cf.map("uid", uid);
        jdbcTemplate(uid).update(sql, parameters);
    }

    @Override
    public void unblock(DjfsUid uid) {
        String sql = collectStats(uid) + " UPDATE disk.user_index SET blocked = false WHERE uid = :uid";
        Map<String, Object> parameters = Cf.map("uid", uid);
        jdbcTemplate(uid).update(sql, parameters);
    }

    @Override
    public void addCollection(DjfsUid uid, String collection) {
        String sql = collectStats(uid) + " UPDATE disk.user_index SET collections = collections || :collection::text "
                + " WHERE uid = :uid AND :collection != ALL (collections)";
        Map<String, Object> parameters = Cf.map("uid", uid, "collection", collection);
        jdbcTemplate(uid).update(sql, parameters);
    }

    @Override
    public void setQuickMoveFlag(DjfsUid uid) {
        String sql = collectStats(uid) + " UPDATE disk.user_index SET is_reindexed_for_quick_move = true WHERE uid = :uid";
        Map<String, Object> parameters = Cf.map("uid", uid);
        jdbcTemplate(uid).update(sql, parameters);
    }

    @Override
    public Option<Long> getFaceClustersVersion(DjfsUid uid) {
        String sql = collectStats(uid) + " SELECT face_clusters_version FROM disk.albums_info WHERE uid = :uid";
        Map<String, Object> parameters = Cf.map("uid", uid);
        return
            jdbcTemplate(uid)
            .queryForOption(sql, (rs, rowNum) -> ResultSetUtils.getLongO(rs,"face_clusters_version"), parameters)
            .getOrElse(Option.empty());
    }

    @Override
    public void updateFaceClustersVersion(DjfsUid uid, long version) {
        String sql =
            collectStats(uid)
            + " INSERT INTO disk.albums_info (uid, revision, face_clusters_version) "
            + "VALUES (:uid, 0, :face_clusters_version) "
            + "ON CONFLICT (uid) DO UPDATE SET face_clusters_version = :face_clusters_version";
        Map<String, Object> parameters = Cf.map("uid", uid, "face_clusters_version", version);
        jdbcTemplate(uid).update(sql, parameters);
    }

    @Override
    public void removeFaceClustersVersion(DjfsUid uid) {
        String sql =
            collectStats(uid)
            + " UPDATE disk.albums_info SET face_clusters_version = NULL WHERE uid = :uid";
        Map<String, Object> parameters = Cf.map("uid", uid);
        jdbcTemplate(uid).update(sql, parameters);
    }

    @Override
    public Tuple2<FacesIndexingState, Instant> getFacesIndexingState(DjfsUid uid) {
        String sql =
            collectStats(uid)
            + " SELECT faces_indexing_state, faces_indexing_state_time FROM disk.user_index WHERE uid = :uid;";
        Map<String, Object> parameters = Cf.map("uid", uid);
        return
            jdbcTemplate(uid)
            .queryForOption(
                sql,
                (rs, rowNum) ->
                    Tuple2.tuple(
                        ResultSetUtils.getStringO(rs, "faces_indexing_state")
                            .flatMapO(FacesIndexingState.R::fromValueO)
                            .getOrElse(FacesIndexingState.NOT_INDEXED),
                        ResultSetUtils.getInstantO(rs, "faces_indexing_state_time").getOrElse(Instant.now())
                    ),
                parameters
            )
            .getOrElse(Tuple2.tuple(FacesIndexingState.NOT_INDEXED, Instant.now()));
    }

    @Override
    public void setFacesIndexingState(DjfsUid uid, FacesIndexingState state) {
        String sql =
            collectStats(uid)
            + " UPDATE disk.user_index"
            + " SET faces_indexing_state = :faces_indexing_state, faces_indexing_state_time = :now WHERE uid = :uid;";
        Map<String, Object> parameters =
            Cf.map("uid", uid, "faces_indexing_state", state, "now", Instant.now());
        jdbcTemplate(uid).update(sql, parameters);
    }
}
