package ru.yandex.chemodan.app.djfs.core.user;

import lombok.Builder;
import lombok.Data;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.db.DjfsUidSource;
import ru.yandex.chemodan.util.exception.BadRequestException;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

@Builder
@Data
public class UserData implements DjfsUidSource {

    public static final String USER_OBJ = "userObj";

    public static UserData cons(DjfsUid uid, UserType type) {
        return new UserData(uid, Option.of(Instant.now()), Option.of(UserLocale.RU), 0, Option.empty(), Option.of(type),
                Option.empty(), Option.empty(), false, Option.empty(), Option.empty(), false, false, Option.empty(),
                Option.empty(), false);
    }

    public static UserData getFromRequest(HttpServletRequestX req) {
        Option<UserData> user = Option.ofNullable(req.getAttribute(USER_OBJ)).filter(UserData.class::isInstance)
                .map(UserData.class::cast);
        return user.getOrThrow(() -> new BadRequestException("No user object in request"));
    }

    public static UserData cons(DjfsUid uid) {
        return cons(uid, UserType.STANDARD);
    }

    private final DjfsUid id;
    private final Option<Instant> regTime;
    private final Option<UserLocale> locale;
    private final int shardKey;
    private final ListF<String> collections;
    private final Option<UserType> type;
    private final Option<Long> version;
    private final Option<Long> minimumDeltaVersion;
    private final boolean blocked;
    private final Option<Instant> deleted;
    private final Option<String> b2bKey;
    private final boolean isQuickMoveUser;
    private final boolean isPg;
    private final Option<String> yateamUid;
    private final Option<Pdd> pdd;
    private final boolean isPaid;

    public boolean isB2b() {
        return b2bKey.filter(StringUtils::isNotBlank).isPresent();
    }

    public User toUser() {
        User user = new User();
        user.id = id;
        user.regTime = regTime;
        user.locale = locale;
        user.shardKey = shardKey;
        user.collections = collections;
        user.type = type;
        user.version = version;
        user.minimumDeltaVersion = minimumDeltaVersion;
        user.blocked = blocked;
        user.deleted = deleted;
        user.b2bKey = b2bKey;
        user.isQuickMoveUser = isQuickMoveUser;
        user.isPg = isPg;
        user.yateamUid = yateamUid;
        return user;
    }

    @Override
    public DjfsUid getUid() {
        return id;
    }
}
