package ru.yandex.chemodan.app.djfs.core.user;

import java.util.concurrent.atomic.AtomicBoolean;

import com.mongodb.ReadPreference;
import lombok.Builder;
import lombok.Value;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.djfs.core.DjfsException;
import ru.yandex.chemodan.app.djfs.core.album.AlbumDao;
import ru.yandex.chemodan.app.djfs.core.album.AlbumFaceClustersDao;
import ru.yandex.chemodan.app.djfs.core.album.AlbumItemDao;
import ru.yandex.chemodan.app.djfs.core.billing.BillingOrderDao;
import ru.yandex.chemodan.app.djfs.core.billing.BillingOrderHistoryDao;
import ru.yandex.chemodan.app.djfs.core.billing.BillingServiceAttributeDao;
import ru.yandex.chemodan.app.djfs.core.billing.BillingServiceAttributeHistoryDao;
import ru.yandex.chemodan.app.djfs.core.billing.BillingServiceDao;
import ru.yandex.chemodan.app.djfs.core.billing.BillingServiceHistoryDao;
import ru.yandex.chemodan.app.djfs.core.billing.BillingSubscriptionDao;
import ru.yandex.chemodan.app.djfs.core.changelog.ChangelogDao;
import ru.yandex.chemodan.app.djfs.core.db.mongo.DjfsBenderFactory;
import ru.yandex.chemodan.app.djfs.core.db.pg.SharpeiShardResolver;
import ru.yandex.chemodan.app.djfs.core.diskinfo.DiskInfoManager;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsPrincipal;
import ru.yandex.chemodan.app.djfs.core.filesystem.Filesystem;
import ru.yandex.chemodan.app.djfs.core.filesystem.exception.ResourceExistsException;
import ru.yandex.chemodan.app.djfs.core.filesystem.iteration.ResourceIterator;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourcePath;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FileDjfsResource;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.FolderDjfsResource;
import ru.yandex.chemodan.app.djfs.core.lock.LockManager;
import ru.yandex.chemodan.app.djfs.core.operations.MpfsOperationHandler;
import ru.yandex.chemodan.app.djfs.core.operations.MpfsOperationHandlerContext;
import ru.yandex.chemodan.app.djfs.core.operations.Operation;
import ru.yandex.chemodan.app.djfs.core.publication.LinkDataDao;
import ru.yandex.chemodan.app.djfs.core.share.GroupInviteDao;
import ru.yandex.chemodan.app.djfs.core.share.ShareManager;
import ru.yandex.chemodan.app.djfs.core.util.UuidUtils;
import ru.yandex.chemodan.app.djfs.core.versioning.VersionManager;
import ru.yandex.chemodan.mpfs.MpfsCallbackResponse;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.util.sharpei.ShardUserInfo;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.misc.bender.BenderParserSerializer;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.net.LocalhostUtils;

/**
 * @author eoshch
 */
public class UserPermanentDeleteOperationHandler extends MpfsOperationHandler {
    private static final Logger logger = LoggerFactory.getLogger(UserPermanentDeleteOperationHandler.class);

    public static final TaskId TASK_ID =
            new TaskId("mpfs.core.job_handlers.operation.handle_user_permanent_delete_operation");
    public final DynamicProperty<Boolean> failTask =
            new DynamicProperty<Boolean>("djfs-worker.fail-user-perm-delete", false);

    public static final String TYPE = "user";
    public static final String SUBTYPE = "permanent_delete";

    private final SharpeiShardResolver sharpeiShardResolver;
    private final UserDao userDao;
    private final MongoInviteMpfsReferralDao mongoInviteMpfsReferralDao;
    private final MongoInviteMpfsSentDao mongoInviteMpfsSentDao;
    private final AlbumDao albumDao;
    private final AlbumFaceClustersDao albumFaceClustersDao;
    private final AlbumItemDao albumItemDao;
    private final BillingOrderDao billingOrderDao;
    private final BillingOrderHistoryDao billingOrderHistoryDao;
    private final BillingServiceAttributeDao billingServiceAttributeDao;
    private final BillingServiceAttributeHistoryDao billingServiceAttributeHistoryDao;
    private final BillingServiceDao billingServiceDao;
    private final BillingServiceHistoryDao billingServiceHistoryDao;
    private final BillingSubscriptionDao billingSubscriptionDao;
    private final ChangelogDao changelogDao;
    private final Filesystem filesystem;
    private final ResourceIterator.Factory resourceIteratorFactory;
    private final LinkDataDao linkDataDao;
    private final ShareManager shareManager;
    private final GroupInviteDao groupInviteDao;
    private final DiskInfoManager diskInfoManager;
    private final LockManager lockManager;
    private final VersionManager versionManager;
    private final MpfsClient mpfsClient;
    private final Blackbox2 blackbox;

    public UserPermanentDeleteOperationHandler(MpfsOperationHandlerContext mpfsOperationHandlerContext,
            SharpeiShardResolver sharpeiShardResolver,
            UserDao userDao, MongoInviteMpfsReferralDao mongoInviteMpfsReferralDao,
            MongoInviteMpfsSentDao mongoInviteMpfsSentDao, AlbumDao albumDao, AlbumFaceClustersDao albumFaceClustersDao,
            AlbumItemDao albumItemDao,
            BillingOrderDao billingOrderDao, BillingOrderHistoryDao billingOrderHistoryDao,
            BillingServiceAttributeDao billingServiceAttributeDao,
            BillingServiceAttributeHistoryDao billingServiceAttributeHistoryDao,
            BillingServiceDao billingServiceDao, BillingServiceHistoryDao billingServiceHistoryDao,
            BillingSubscriptionDao billingSubscriptionDao, ChangelogDao changelogDao,
            Filesystem filesystem, ResourceIterator.Factory resourceIteratorFactory, LinkDataDao linkDataDao,
            ShareManager shareManager, GroupInviteDao groupInviteDao, DiskInfoManager diskInfoManager,
            LockManager lockManager, VersionManager versionManager, MpfsClient mpfsClient, Blackbox2 blackbox)
    {
        super(mpfsOperationHandlerContext);
        this.sharpeiShardResolver = sharpeiShardResolver;
        this.userDao = userDao;
        this.mongoInviteMpfsReferralDao = mongoInviteMpfsReferralDao;
        this.mongoInviteMpfsSentDao = mongoInviteMpfsSentDao;
        this.albumDao = albumDao;
        this.albumFaceClustersDao = albumFaceClustersDao;
        this.albumItemDao = albumItemDao;
        this.billingOrderDao = billingOrderDao;
        this.billingOrderHistoryDao = billingOrderHistoryDao;
        this.billingServiceAttributeDao = billingServiceAttributeDao;
        this.billingServiceAttributeHistoryDao = billingServiceAttributeHistoryDao;
        this.billingServiceDao = billingServiceDao;
        this.billingServiceHistoryDao = billingServiceHistoryDao;
        this.billingSubscriptionDao = billingSubscriptionDao;
        this.changelogDao = changelogDao;
        this.filesystem = filesystem;
        this.resourceIteratorFactory = resourceIteratorFactory;
        this.linkDataDao = linkDataDao;
        this.shareManager = shareManager;
        this.groupInviteDao = groupInviteDao;
        this.diskInfoManager = diskInfoManager;
        this.lockManager = lockManager;
        this.versionManager = versionManager;
        this.mpfsClient = mpfsClient;
        this.blackbox = blackbox;
    }

    @Override
    protected TaskId celeryTaskId() {
        return TASK_ID;
    }

    @Override
    public Duration timeout() {
        return Duration.standardDays(1);
    }

    @Override
    protected Status handle(Operation operation, AtomicBoolean terminated) {
        DjfsUid uid = operation.getUid();

        if (failTask.get()) {
            logger.error("Force fail mpfs.core.job_handlers.operation.handle_user_permanent_delete_operation id = {}",
                    operation.getId());
            return Status.FAIL;
        }

        if (!isHandlingNeeded(uid)) {
            return Status.DONE;
        }

        if (lockManager.isLocked(uid)) {
            logger.info("uid " + uid.asString() + " locked. delayed.");
            return Status.DELAY;
        }

        ensureMandatoryDataIsPresent(uid);

        logger.info("deleting non-file data of " + uid.asString());
        deleteGeneralUserData(uid);
        deleteBillingData(uid);
        deleteShareFoldersData(uid);

        logger.info("moving all files of " + uid.asString() + " to hidden");
        moveFilesToHiddenData(operation);
        deleteFilesystemMetadata(uid);

        return Status.DONE;
    }

    public boolean isHandlingNeeded(DjfsUid uid) {
        Option<ShardUserInfo> shardUserInfo = sharpeiShardResolver.shardByUid(uid);
        if (!shardUserInfo.isPresent()) {
            logger.info(uid.asString() + " is not initialized. skipping.");
            return false;
        }

        BlackboxCorrectResponse response = blackbox.query().userInfo(LocalhostUtils.localAddress(),
                uid.asPassportUid(), Cf.<String>list());

        if (response.getUid().isPresent()) {
            logger.warn(uid.asString() + " is found in passport. skipping.");
            return false;
        }

        logger.info(uid.asString() + " is not found in passport. ok to delete.");
        return true;
    }

    private void ensureMandatoryDataIsPresent(DjfsUid uid) {
        Option<UserData> userO = userDao.find(uid);
        if (!userO.isPresent()) {
            userDao.insert(UserData.cons(uid, UserType.STANDARD));
        } else {
            UserData user = userO.get();
            if (user.getType().getOrNull() != UserType.STANDARD) {
                logger.info("{} type is {}. changing to standard.", uid.asString(), user.getType());
                userDao.changeType(uid, UserType.STANDARD);
            }
            if (user.isBlocked()) {
                logger.info("{} is blocked. unblocking to delete all data.", uid.asString());
                userDao.unblock(uid);
            }
        }

        diskInfoManager.ensureRootExists(uid);
        diskInfoManager.ensureLimitExists(uid);
        userDao.addCollection(uid, "disk_info");
    }

    private void deleteGeneralUserData(DjfsUid uid) {
        userDao.setDeleted(uid);
        mongoInviteMpfsSentDao.deleteAll(uid);
        mongoInviteMpfsReferralDao.delete(uid);
        mongoInviteMpfsReferralDao.deleteAll(uid);
    }

    private void deleteBillingData(DjfsUid uid) {
        billingOrderDao.deleteAll(uid);
        billingOrderHistoryDao.deleteAll(uid);
        billingServiceAttributeDao.deleteAll(uid);
        billingServiceAttributeHistoryDao.deleteAll(uid);
        billingServiceDao.deleteAll(uid);
        billingServiceHistoryDao.deleteAll(uid);
        billingSubscriptionDao.deleteAll(uid);
    }

    private void deleteShareFoldersData(DjfsUid uid) {
        groupInviteDao.deleteAll(uid);
        shareManager.removeAllSharedFolders(DjfsPrincipal.SYSTEM, uid);
    }

    private void moveFilesToHiddenData(Operation operation) {
        DjfsUid uid = operation.getUid();
        Data data = operation.getData(Data.B);
        if (!data.hiddenRootFolderName.isPresent()) {
            String hiddenRootFolderName = "user-deleted-root-" + Instant.now().toString() + "-"
                    + UuidUtils.randomToHexString();
            data = Data.builder().hiddenRootFolderName(Option.of(hiddenRootFolderName)).build();
            operationDao.setData(uid, operation.getId(), data, Data.B, operation);
        }

        DjfsResourcePath hiddenDataTargetRoot = DjfsResourcePath.areaRoot(uid, DjfsResourceArea.HIDDEN)
                .getChildPath(data.hiddenRootFolderName.get());
        try {
            filesystem.createFolder(DjfsPrincipal.SYSTEM, hiddenDataTargetRoot,
                    x -> x.hiddenAppendTime(Option.of(Instant.now())));
        } catch (ResourceExistsException e) {
            // we want to be sure folder exists
        }

        for (DjfsResourceArea area : Cf.wrap(DjfsResourceArea.values()).filter(x -> x != DjfsResourceArea.HIDDEN)) {
            DjfsResourcePath iterationRoot = DjfsResourcePath.areaRoot(uid, area);
            Option<DjfsResource> iterationRootResource =
                    filesystem.find(DjfsPrincipal.SYSTEM, iterationRoot, Option.of(ReadPreference.primary()));
            if (iterationRootResource.isPresent() && !iterationRootResource.get().getResourceId().isPresent()) {
                filesystem.ensureFileIdIsSet(DjfsPrincipal.SYSTEM, iterationRoot);
            }

            ResourceIterator iterator = resourceIteratorFactory.create(iterationRoot,
                    ResourceIterator.TraversalType.BOTTOM_UP);
            ResourceIterator.ResourceIteratorState state = iterator.initialize();
            while (iterator.hasNext(state)) {
                Tuple2<ResourceIterator.ResourceIteratorState, ListF<DjfsResource>> next = iterator.next(state, 10);
                state = next._1;
                for (DjfsResource resource : next._2) {
                    if (resource instanceof FolderDjfsResource) {
                        DjfsResourcePath newPath = resource.getPath().changeParent(
                                DjfsResourcePath.root(uid, area), hiddenDataTargetRoot);
                        filesystem.ensureAllParentFoldersExist(DjfsPrincipal.SYSTEM, newPath);

                        try {
                            filesystem.createFolder(DjfsPrincipal.SYSTEM, newPath);
                        } catch (ResourceExistsException e) {
                            // we want to be sure folder exists
                        }

                        MpfsCallbackResponse response = mpfsClient.rm(uid.asMpfsUser(), resource.getPath().getPath());
                        if (response.getStatusCode() < 200 || response.getStatusCode() > 299) {
                            throw new DjfsException("mpfs rm call failed: " + response.toString());
                        }
                    }
                    if (resource instanceof FileDjfsResource) {
                        DjfsResourcePath newPath = resource.getPath().changeParent(
                                DjfsResourcePath.root(uid, area), hiddenDataTargetRoot);
                        filesystem.ensureAllParentFoldersExist(DjfsPrincipal.SYSTEM, newPath);
                        filesystem.moveToHidden(DjfsPrincipal.SYSTEM, (FileDjfsResource) resource, newPath);
                        versionManager.removeVersions(resource.getResourceId().get());
                    }
                }
            }
        }
    }

    private void deleteFilesystemMetadata(DjfsUid uid) {
        albumFaceClustersDao.deleteAll(uid);
        albumDao.deleteAll(uid);
        albumItemDao.deleteAll(uid);
        linkDataDao.deleteAll(uid);
        changelogDao.deleteAll(uid);
    }

    @Value
    @Builder(toBuilder = true)
    @BenderBindAllFields
    public static class Data {
        public static final BenderParserSerializer<Data> B =
                DjfsBenderFactory.createForJson(Data.class);

        @Builder.Default
        private final Option<String> hiddenRootFolderName = Option.empty();
    }
}
