package ru.yandex.chemodan.app.djfs.core.user;

import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.djfs.core.db.pg.SharpeiShardResolver;
import ru.yandex.chemodan.app.djfs.core.filesystem.DjfsPrincipal;
import ru.yandex.chemodan.app.djfs.core.lock.LockManager;
import ru.yandex.chemodan.app.djfs.core.operations.MpfsOperationHandler;
import ru.yandex.chemodan.app.djfs.core.operations.MpfsOperationHandlerContext;
import ru.yandex.chemodan.app.djfs.core.operations.Operation;
import ru.yandex.chemodan.app.djfs.core.share.ShareManager;
import ru.yandex.chemodan.util.sharpei.ShardUserInfo;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxAttributes;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.net.LocalhostUtils;

/**
 * @author eoshch
 */
public class UserPreliminaryDeleteOperationHandler extends MpfsOperationHandler {
    private static final Logger logger = LoggerFactory.getLogger(UserPreliminaryDeleteOperationHandler.class);

    private final SharpeiShardResolver sharpeiShardResolver;
    private final ShareManager shareManager;
    private final LockManager lockManager;
    private final Blackbox2 blackbox;

    public UserPreliminaryDeleteOperationHandler(MpfsOperationHandlerContext mpfsOperationHandlerContext,
            SharpeiShardResolver sharpeiShardResolver,
            ShareManager shareManager, LockManager lockManager, Blackbox2 blackbox)
    {
        super(mpfsOperationHandlerContext);
        this.sharpeiShardResolver = sharpeiShardResolver;
        this.shareManager = shareManager;
        this.lockManager = lockManager;
        this.blackbox = blackbox;
    }

    @Override
    protected TaskId celeryTaskId() {
        return new TaskId("mpfs.core.job_handlers.operation.handle_user_preliminary_delete_operation");
    }

    @Override
    public Duration timeout() {
        return Duration.standardDays(1);
    }

    @Override
    protected Status handle(Operation operation, AtomicBoolean terminated) {
        DjfsUid uid = operation.getUid();

        BlackboxCorrectResponse response = blackbox.query().userInfo(LocalhostUtils.localAddress(),
                uid.asPassportUid(), Cf.list(BlackboxAttributes.ACCOUNT_IS_AVAILABLE));

        if (!response.getUid().isPresent()) {
            logger.info("no info about " + uid.asString() + " in passport. skipping.");
            return Status.DONE;
        }

        Option<String> isAvailable = response.getAttributes().getO(BlackboxAttributes.ACCOUNT_IS_AVAILABLE);
        if (isAvailable.isPresent() && !Objects.equals("0", isAvailable.get())) {
            logger.info("ACCOUNT_IS_AVAILABLE attribute is " + isAvailable.get() + " for " + uid.asString()
                    + " in passport. skipping.");
            return Status.DONE;
        }

        Option<ShardUserInfo> shardUserInfo = sharpeiShardResolver.shardByUid(uid);
        if (!shardUserInfo.isPresent()) {
            logger.info(uid.asString() + " is not initialized. skipping.");
            return Status.DONE;
        }

        if (lockManager.isLocked(uid)) {
            logger.info("uid " + uid.asString() + " locked. delayed.");
            return Status.DELAY;
        }

        shareManager.removeAllSharedFolders(DjfsPrincipal.SYSTEM, uid);
        return Status.DONE;
    }
}
