package ru.yandex.chemodan.app.djfs.core.web;

import lombok.RequiredArgsConstructor;

import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.a3.action.CloneableAction;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.intercept.ActionInvocationInterceptor;
import ru.yandex.commune.a3.action.intercept.InvocationInterceptorOrders;
import ru.yandex.commune.a3.action.invoke.ActionInvocation;
import ru.yandex.commune.a3.action.invoke.ActionMethodInvocation;
import ru.yandex.commune.a3.action.invoke.CloneableActionInvocation;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicySource;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@RequiredArgsConstructor
public class DynamicMasterSlavePolicyInterceptor implements ActionInvocationInterceptor {
    private static final Logger log = LoggerFactory.getLogger(DynamicMasterSlavePolicyInterceptor.class);
    private final MasterSlavePolicyOverridesRegistry masterSlavePolicyOverridesRegistry;

    @Override
    public Object intercept(ActionInvocation invocation) throws Exception {
        MasterSlavePolicy policy = null;

        Option<WithMasterSlavePolicy> wmsp =
                invocation.getActionDescriptor().getAnnotationO(WithMasterSlavePolicy.class);
        if (wmsp.isPresent()) {
            policy = wmsp.get().value();
        } else if (invocation instanceof CloneableActionInvocation) {
            CloneableAction action = ((CloneableActionInvocation) invocation).getAction();
            if (action instanceof MasterSlavePolicySource) {
                policy = ((MasterSlavePolicySource) action).getMasterSlavePolicy();
            }
        } else if (invocation instanceof ActionMethodInvocation) {
            Option<WithMasterSlavePolicy> annotation = Option.ofNullable(
                    invocation.getActionDescriptor().getContainerClass().getAnnotation(WithMasterSlavePolicy.class));
            if (annotation.isPresent()) {
                policy = annotation.get().value();
            }
        }

        if (!masterSlavePolicyOverridesRegistry.getAll().isEmpty()) {
            Option<MasterSlavePolicy> policyOverride = getPolicyOverride(invocation);
            if (policyOverride.isPresent()) {
                policy = policyOverride.get();
                log.info("Master-slave policy overrides to {}", policy);
            }
        }

        if (policy != null) {
            MasterSlaveContextHolder.PolicyHandle old = MasterSlaveContextHolder.push(policy);
            log.info("Using master-slave policy: {}", policy);
            try {
                return invocation.invoke();
            } finally {
                old.popSafely();
            }
        } else {
            return invocation.invoke();
        }
    }

    @Override
    public int getOrder() {
        return InvocationInterceptorOrders.MASTER_SLAVE_INTERCEPTOR_ORDER;
    }

    private Option<MasterSlavePolicy> getPolicyOverride(ActionInvocation invocation) {
        Option<Path> pathO = invocation.getActionDescriptor().getAnnotationO(Path.class);
        if (!pathO.isPresent()) {
            return Option.empty();
        }
        Path path = pathO.get();
        MasterSlavePolicyOverridesRegistry.PathDefinition def = new MasterSlavePolicyOverridesRegistry.PathDefinition(
                path.value(),
                HttpMethod.R.valueOfO(invocation.getWebRequest().getHttpServletRequest().getMethod())
        );

        Option<MasterSlavePolicyOverridesRegistry.MasterSlavePolicyOverride> override =
                masterSlavePolicyOverridesRegistry.getO(def);
        if (override.isPresent()) {
            return override.map(MasterSlavePolicyOverridesRegistry.MasterSlavePolicyOverride::getPolicy);
        }

        override = masterSlavePolicyOverridesRegistry.getO(new MasterSlavePolicyOverridesRegistry.PathDefinition(
                path.value(), Option.empty()
        ));
        return override.map(MasterSlavePolicyOverridesRegistry.MasterSlavePolicyOverride::getPolicy);
    }
}
