package ru.yandex.chemodan.app.djfs.worker;

import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import lombok.Builder;
import lombok.Value;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.db.DjfsShardInfo;
import ru.yandex.chemodan.app.djfs.core.db.mongo.DjfsBenderFactory;
import ru.yandex.chemodan.app.djfs.core.filesystem.MongoDjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.PgDjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.StidType;
import ru.yandex.chemodan.app.djfs.core.operations.MpfsOperationHandler;
import ru.yandex.chemodan.app.djfs.core.operations.MpfsOperationHandlerContext;
import ru.yandex.chemodan.app.djfs.core.operations.Operation;
import ru.yandex.chemodan.app.djfs.core.operations.OperationDao;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.CeleryJobUtils;
import ru.yandex.chemodan.app.djfs.core.util.DjfsAsyncTaskUtils;
import ru.yandex.chemodan.app.djfs.core.util.JsonUtils;
import ru.yandex.chemodan.queller.celery.job.CeleryJob;
import ru.yandex.chemodan.queller.worker.CeleryTaskManager;
import ru.yandex.chemodan.util.yt.YtHelper;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.json.JsonValue;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryType;
import ru.yandex.inside.yt.kosher.tables.types.JacksonTableEntryType;
import ru.yandex.misc.bender.BenderParserSerializer;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * Launch from script:
 * <p>
 * var s = java.util.UUID.fromString("00000000-0000-0000-0000-000000000000");
 * var e = java.util.UUID.fromString("00000000-0000-0000-0000-100000000000");
 * exportOctober2018StorageFailureStidCandidatesToYtOperationSender.send("mongo", "disk-unit-14", "hidden", s, e);
 *
 * @author eoshch
 */
public interface ExportOctober2018StorageFailureStidCandidatesToYtOperation {
    TaskId TASK_ID = new TaskId("handle_export_october_2018_storage_failure_stid_candidates_to_yt_operation");

    String TYPE = "auxiliary";
    String SUBTYPE = "export_october_2018_storage_failure_stid_candidates_to_yt";

    @Value
    @Builder(toBuilder = true)
    @BenderBindAllFields
    class Data {
        public static final BenderParserSerializer<Data> B = DjfsBenderFactory.createForJson(Data.class);

        private final String databaseType;
        private final String shardId;
        private final String area;

        @Builder.Default
        private final Option<UUID> start = Option.empty();

        @Builder.Default
        private final Option<UUID> end = Option.empty();

        @Builder.Default
        private final Option<UUID> current = Option.empty();
    }

    class Sender {
        private static final Logger logger = LoggerFactory.getLogger(Sender.class);

        private final CeleryTaskManager celeryTaskManager;
        private final OperationDao operationDao;

        public Sender(CeleryTaskManager celeryTaskManager, OperationDao operationDao) {
            this.celeryTaskManager = celeryTaskManager;
            this.operationDao = operationDao;
        }

        public void sendPortionOfAllShards(UUID start, UUID end) {
            ListF<String> pgShards = Cf.list("119", "1", "37", "3", "70", "71", "107", "108", "109", "110", "111",
                    "112", "105", "106", "115", "116", "117", "118", "120", "122", "123", "121", "157", "156", "158",
                    "159", "161", "162", "163", "164", "166", "165", "167");

            for (String shard : pgShards) {
                send("pg", shard, "disk", Option.of(start), Option.of(end));
            }
        }

        public void send(String databaseType, String shardId, String area, UUID start, UUID end) {
            send(databaseType, shardId, area, Option.of(start), Option.of(end));
        }

        public void send(String databaseType, String shardId, String area, Option<UUID> start, Option<UUID> end) {
            DjfsUid uid = DjfsUid.cons(739088530);
            // magical uid: 739088530

            Data data = Data.builder()
                    .databaseType(databaseType)
                    .shardId(shardId)
                    .area(area)
                    .start(start)
                    .end(end)
                    .build();

            Operation operation = Operation.cons(uid, TYPE, SUBTYPE, data, Data.B);
            operationDao.insert(operation);
            logger.info("created ExportOctober2018StorageFailureStidCandidatesToYtOperation " + operation.getId()
                    + " for " + uid.asString());

            MapF<String, JsonValue> kwargs = JsonUtils.objectBuilder()
                    .add("uid", uid.asString())
                    .add("oid", operation.getId()).toMap();
            String activeUid = DjfsAsyncTaskUtils.activeUid(uid.asString() + ":" + operation.getId());
            CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, activeUid, kwargs);
            celeryTaskManager.submit(celeryJob);
        }
    }

    class Handler extends MpfsOperationHandler {
        private static final Logger logger = LoggerFactory.getLogger(Handler.class);

        private static final SetF<String> affectedCouples = Cf.set("E1402077:", "E1405880:", "E1406745:", "E217957:",
                "E985779:", "E1219441:", "E1220779:", "E1220947:", "E1245195:", "E1245531:", "E1358783:", "E1361384:",
                "E1392621:", "E1475327:", "E1510702:", "E494525:", "E985928:", "E170242:");

        private static YPath yPath = YPath.simple("//home/mpfs-stat/tmp/storage-failure-stid-candidates-10-2018")
                .append(true);
        private static YTableEntryType<JsonNode> yTableEntryType = new JacksonTableEntryType();

        private final YtHelper yt;
        private final MongoDjfsResourceDao mongoDjfsResourceDao;
        private final PgDjfsResourceDao pgDjfsResourceDao;

        public Handler(YtHelper yt, MpfsOperationHandlerContext mpfsOperationHandlerContext,
                MongoDjfsResourceDao mongoDjfsResourceDao, PgDjfsResourceDao pgDjfsResourceDao)
        {
            super(mpfsOperationHandlerContext);
            this.yt = yt;
            this.mongoDjfsResourceDao = mongoDjfsResourceDao;
            this.pgDjfsResourceDao = pgDjfsResourceDao;
        }

        @Override
        protected Status handle(Operation operation, AtomicBoolean terminated) {
            Data data = operation.getData(Data.B);

            ListF<Tuple3<UUID, StidType, String>> candidateStids = Cf.arrayList();
            int filtered = 0;

            while (true) {
                ListF<Tuple3<UUID, StidType, String>> stids;
                if (Objects.equals(data.databaseType, "mongo")) {
                    DjfsShardInfo.Mongo shardInfo = new DjfsShardInfo.Mongo(data.shardId);
                    stids = mongoDjfsResourceDao.findStids(shardInfo, DjfsResourceArea.R.fromValue(data.area),
                            data.current.orElse(data.start), data.end, 1000);
                } else if (Objects.equals(data.databaseType, "pg")) {
                    DjfsShardInfo.Pg shardInfo = new DjfsShardInfo.Pg(Integer.parseInt(data.shardId));
                    stids = pgDjfsResourceDao.findStids(shardInfo, data.current.orElse(data.start), data.end, 1000);
                } else {
                    throw new NotImplementedException("unknown database type " + data.databaseType);
                }

                if (stids.isEmpty()) {
                    uploadStids(candidateStids, data.shardId);
                    return Status.DONE;
                }

                data = data.toBuilder().current(Option.of(stids.last()._1)).build();
                filtered += stids.length();
                candidateStids.addAll(stids.filter(x -> affectedCouples.filter(y -> x._3.contains(y)).isNotEmpty()));

                if (filtered > 100000 || candidateStids.length() > 100000) {
                    uploadStids(candidateStids, data.shardId);
                    candidateStids = Cf.arrayList();
                    filtered = 0;
                    operationDao.setData(operation.getUid(), operation.getId(), data, Data.B, operation);
                }
            }
        }

        private void uploadStids(ListF<Tuple3<UUID, StidType, String>> stids, String shardId) {
            yt.runWithRetries(() -> yt.tables().write(yPath, yTableEntryType, stids.map(x ->
                    JsonNodeFactory.instance.objectNode()
                            .put("stid", x._3)
                            .put("type", x._2.value())
                            .put("shard", shardId)
            )));
        }

        @Override
        protected TaskId celeryTaskId() {
            return TASK_ID;
        }

        @Override
        public Duration timeout() {
            return Duration.standardDays(1);
        }
    }
}
