package ru.yandex.chemodan.app.djfs.worker;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;

import lombok.Builder;
import lombok.Value;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.djfs.core.db.DjfsShardInfo;
import ru.yandex.chemodan.app.djfs.core.db.mongo.DjfsBenderFactory;
import ru.yandex.chemodan.app.djfs.core.filesystem.MongoDjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.PgDjfsResourceDao;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.DjfsResourceArea;
import ru.yandex.chemodan.app.djfs.core.filesystem.model.StidType;
import ru.yandex.chemodan.app.djfs.core.operations.MpfsOperationHandler;
import ru.yandex.chemodan.app.djfs.core.operations.MpfsOperationHandlerContext;
import ru.yandex.chemodan.app.djfs.core.operations.Operation;
import ru.yandex.chemodan.app.djfs.core.operations.OperationDao;
import ru.yandex.chemodan.app.djfs.core.user.DjfsUid;
import ru.yandex.chemodan.app.djfs.core.util.CeleryJobUtils;
import ru.yandex.chemodan.app.djfs.core.util.DjfsAsyncTaskUtils;
import ru.yandex.chemodan.app.djfs.core.util.JsonUtils;
import ru.yandex.chemodan.queller.celery.job.CeleryJob;
import ru.yandex.chemodan.queller.worker.CeleryTaskManager;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.json.JsonValue;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.BenderParserSerializer;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.yql.except.YqlException;

/**
 * Launch from script:
 * <p>
 * var s = java.util.UUID.fromString("00000000-0000-0000-0000-000000000000");
 * var e = java.util.UUID.fromString("00000000-0000-0000-0000-100000000000");
 * exportStidsToYtOperationSender.send("mongo", "disk-unit-14", "hidden", s, e);
 *
 * @author eoshch
 */
public interface ExportStidsToYtOperation {
    TaskId TASK_ID = new TaskId("handle_export_stids_to_yt_operation");

    String TYPE = "auxiliary";
    String SUBTYPE = "export_stids_to_yt";

    @Value
    @Builder(toBuilder = true)
    @BenderBindAllFields
    class Data {
        public static final BenderParserSerializer<Data> B = DjfsBenderFactory.createForJson(Data.class);

        private final String databaseType;
        private final String shardId;
        private final String area;

        @Builder.Default
        private final Option<UUID> start = Option.empty();

        @Builder.Default
        private final Option<UUID> end = Option.empty();

        @Builder.Default
        private final Option<UUID> current = Option.empty();
    }

    class Sender {
        private static final Logger logger = LoggerFactory.getLogger(Sender.class);

        private final CeleryTaskManager celeryTaskManager;
        private final OperationDao operationDao;

        public Sender(CeleryTaskManager celeryTaskManager, OperationDao operationDao) {
            this.celeryTaskManager = celeryTaskManager;
            this.operationDao = operationDao;
        }

        public void send(String databaseType, String shardId, String area, UUID start, UUID end) {
            send(databaseType, shardId, area, Option.of(start), Option.of(end));
        }

        public void send(String databaseType, String shardId, String area, Option<UUID> start, Option<UUID> end) {
            DjfsUid uid = DjfsUid.cons(739088530);
            // magical uid: 739088530

            Data data = Data.builder()
                    .databaseType(databaseType)
                    .shardId(shardId)
                    .area(area)
                    .start(start)
                    .end(end)
                    .build();

            Operation operation = Operation.cons(uid, TYPE, SUBTYPE, data, Data.B);
            operationDao.insert(operation);
            logger.info("created ExportStidsToYtOperation " + operation.getId() + " for " + uid.asString());

            MapF<String, JsonValue> kwargs = JsonUtils.objectBuilder()
                    .add("uid", uid.asString())
                    .add("oid", operation.getId()).toMap();
            String activeUid = DjfsAsyncTaskUtils.activeUid(uid.asString() + ":" + operation.getId());
            CeleryJob celeryJob = CeleryJobUtils.create(TASK_ID, activeUid, kwargs);
            celeryTaskManager.submit(celeryJob);
        }
    }

    class Handler extends MpfsOperationHandler {
        private static final Logger logger = LoggerFactory.getLogger(Handler.class);

        private final String yqlUrl;
        private final String yqlToken;

        private final MongoDjfsResourceDao mongoDjfsResourceDao;
        private final PgDjfsResourceDao pgDjfsResourceDao;

        public Handler(String yqlUrl, String yqlToken, MpfsOperationHandlerContext mpfsOperationHandlerContext,
                MongoDjfsResourceDao mongoDjfsResourceDao, PgDjfsResourceDao pgDjfsResourceDao)
        {
            super(mpfsOperationHandlerContext);
            this.yqlUrl = yqlUrl;
            this.yqlToken = yqlToken;
            this.mongoDjfsResourceDao = mongoDjfsResourceDao;
            this.pgDjfsResourceDao = pgDjfsResourceDao;
        }

        @Override
        protected Status handle(Operation operation, AtomicBoolean terminated) {
            Data data = operation.getData(Data.B);

            while (true) {
                ListF<Tuple3<UUID, StidType, String>> stids;
                if (Objects.equals(data.databaseType, "mongo")) {
                    DjfsShardInfo.Mongo shardInfo = new DjfsShardInfo.Mongo(data.shardId);
                    stids = mongoDjfsResourceDao.findStids(shardInfo, DjfsResourceArea.R.fromValue(data.area),
                            data.current.orElse(data.start), data.end, 1000);
                } else if (Objects.equals(data.databaseType, "pg")) {
                    DjfsShardInfo.Pg shardInfo = new DjfsShardInfo.Pg(Integer.parseInt(data.shardId));
                    stids = pgDjfsResourceDao.findStids(shardInfo, data.current.orElse(data.start), data.end, 1000);
                } else {
                    throw new NotImplementedException("unknown database type " + data.databaseType);
                }

                if (stids.isEmpty()) {
                    return Status.DONE;
                }

                String yql = "INSERT INTO [//home/mpfs-stat/tmp/all-stids-10-2018] ('stid', 'type', 'shard') "
                        + " VALUES " + StringUtils.join(stids.map(x -> "(?, ?, ?)"), ", ");

                try (Connection connection = DriverManager.getConnection(yqlUrl, "unused", yqlToken);
                     PreparedStatement statement = connection.prepareStatement(yql))
                {
                    int i = 1;
                    for (Tuple3<UUID, StidType, String> stid : stids) {
                        statement.setString(i, stid._3);
                        statement.setString(i + 1, stid._2.value());
                        statement.setString(i + 2, data.shardId);
                        i += 3;
                    }
                    statement.executeUpdate();
                } catch (YqlException e) {
                    logger.error("exception, will delay: ", e);
                    return Status.DELAY;
                } catch (SQLException e) {
                    throw ExceptionUtils.translate(e);
                }

                data = data.toBuilder().current(Option.of(stids.last()._1)).build();
                operationDao.setData(operation.getUid(), operation.getId(), data, Data.B, operation);
            }
        }

        @Override
        protected TaskId celeryTaskId() {
            return TASK_ID;
        }

        @Override
        public Duration timeout() {
            return Duration.standardDays(1);
        }
    }
}
