package ru.yandex.chemodan.app.docviewer.adapters.batik;

import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.OutputStream;

import javax.annotation.PostConstruct;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;

import org.apache.batik.svggen.SVGGraphics2D;
import org.apache.batik.transcoder.TranscoderInput;
import org.apache.batik.transcoder.TranscoderOutput;
import org.apache.batik.transcoder.image.PNGTranscoder;
import org.apache.batik.transcoder.wmf.tosvg.WMFTranscoder;
import org.springframework.beans.factory.annotation.Value;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

import ru.yandex.chemodan.app.docviewer.utils.ByteArrayOutputStreamSource;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.IoFunction1V;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.time.Stopwatch;

public class BatikAdapter {

    private static final Logger logger = LoggerFactory.getLogger(BatikAdapter.class);

    private static final char[] symbolToLowercaseGreeks = new char[]{'α', 'β', 'χ', 'δ', 'ε',
            'ϕ', 'γ', 'η', 'ι', 'φ', 'κ', 'λ', 'μ', 'ν', 'ο', 'π', 'θ', 'ρ', 'σ', 'τ', 'υ', 'ϖ',
            'ω', 'ξ', 'ψ', 'ζ'};

    private static final char[] symbolToUppercaseGreeks = new char[]{'Α', 'Β', 'Χ', 'Δ', 'Ε',
            'Φ', 'Γ', 'Η', 'Ι', 'ϑ', 'Κ', 'Λ', 'Μ', 'Ν', 'Ο', 'Π', 'Θ', 'Ρ', 'Σ', 'Τ', 'Υ', 'ς',
            'Ω', 'Ξ', 'Ψ', 'Ζ'};
    private static final Object MONITOR = new Object();
    @Value("${batik.monitor.enabled}")
    private boolean isMonitorEnabled;
    private DocumentBuilderFactory documentBuilderFactory;
    private TransformerFactory transformerFactory;
    private XPathFactory xPathFactory;

    private static void replaceWithGreek(Text textNode) {
        String data = textNode.getData();

        StringBuilder stringBuilder = new StringBuilder();
        for (char c : data.toCharArray()) {
            if ('a' <= c && c <= 'z') {
                c = symbolToLowercaseGreeks[c - 'a'];
            }
            if ('A' <= c && c <= 'Z') {
                c = symbolToUppercaseGreeks[c - 'A'];
            }
            stringBuilder.append(c);
        }

        textNode.setData(stringBuilder.toString());
    }

    @PostConstruct
    public void afterPropertiesSet() {
        documentBuilderFactory = DocumentBuilderFactory.newInstance();
        documentBuilderFactory.setNamespaceAware(true);

        transformerFactory = TransformerFactory.newInstance();

        xPathFactory = XPathFactory.newInstance();
    }

    private void fixSymbolFontIssue(Document document) {
        try {
            XPath xPath = xPathFactory.newXPath();

            NodeList textNodes = (NodeList) xPath.evaluate(
                    "//*[contains(@font-family, 'Symbol')]//text()", document,
                    XPathConstants.NODESET);
            for (int i = 0; i < textNodes.getLength(); i++) {
                Text textNode = (Text) textNodes.item(i);

                boolean hasSymbolParent = false;
                Node parentNode = textNode;
                while (parentNode != null) {

                    if (parentNode.getNodeType() == Node.ELEMENT_NODE) {
                        Element parentElement = (Element) parentNode;
                        String fontFamily = parentElement.getAttribute("font-family");
                        if (StringUtils.isNotEmpty(fontFamily)) {
                            if (StringUtils.equalsIgnoreCase("'Symbol'", fontFamily)) {
                                hasSymbolParent = true;
                            }
                            break;
                        }
                    }

                    parentNode = parentNode.getParentNode();
                }

                if (hasSymbolParent) {
                    replaceWithGreek(textNode);
                }
            }
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
    }

    public void transcodeSvgToPng(final InputStreamSource inputStreamSource,
            final OutputStreamSource outputStreamSource, final float width, final float height)
    {
        inputStreamSource.buffered().readNr(inputStream -> outputStreamSource.writeBuffered(outputStream -> {
            transcodeSvgToPng(new TranscoderInput(inputStream), outputStream, width,
                    height);
        }));
    }

    private void transcodeSvgToPng(TranscoderInput svgInput, OutputStream pngOutput, float width,
            float height)
    {
        Stopwatch stopwatch = Stopwatch.createAndStart();
        try {
            PNGTranscoder pngTranscoder = new PNGTranscoder();
            pngTranscoder.addTranscodingHint(PNGTranscoder.KEY_WIDTH, width);
            pngTranscoder.addTranscodingHint(PNGTranscoder.KEY_HEIGHT, height);

            TranscoderOutput transcoderOutput = new TranscoderOutput(pngOutput);
            if (isMonitorEnabled) {
                // prevent hanging on internal cache https://st.yandex-team.ru/DOCVIEWER-1637
                synchronized (MONITOR) {
                    pngTranscoder.transcode(svgInput, transcoderOutput);
                }
            } else {
                pngTranscoder.transcode(svgInput, transcoderOutput);
            }
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        } finally {
            stopwatch.stopAndLog("Transcode from SVG to PNG", logger);
        }
    }

    public void transcodeWmfToSvg(final InputStreamSource inputStreamSource,
            final OutputStreamSource outputStreamSource)
    {
        inputStreamSource.buffered().readNr(inputStream -> transcodeWmfToSvg(new TranscoderInput(inputStream), outputStreamSource));
    }

    private Document transcodeWmfToSvg(TranscoderInput input) {
        ThreadLocalTimeout.check();

        try {
            Document svgDocument = documentBuilderFactory.newDocumentBuilder().newDocument();
            TranscoderOutput transcoderOutput = new TranscoderOutput(svgDocument);

            final Element[] result = new Element[1];
            WMFTranscoder wmfTranscoder = new WMFTranscoder() {

                @Override
                protected void writeSVGToOutput(SVGGraphics2D pSvgGenerator, Element pSvgRoot,
                        TranscoderOutput pOutput) {
                    result[0] = pSvgRoot;
                }
            };
            wmfTranscoder.transcode(input, transcoderOutput);
            svgDocument.appendChild(result[0]);

            fixSymbolFontIssue(svgDocument);
            return svgDocument;
        } catch (Exception exc) {
            throw ExceptionUtils.translate(exc);
        }
    }

    private void transcodeWmfToSvg(final TranscoderInput input, final OutputStreamSource output) {
        output.writeBuffered((IoFunction1V<BufferedOutputStream>) outputStream -> {
            Document svgDocument = transcodeWmfToSvg(input);
            ThreadLocalTimeout.check();
            Transformer transformer = transformerFactory.newTransformer();
            transformer.transform(new DOMSource(svgDocument), new StreamResult(outputStream));
        });
    }

    byte[] transcodeWmfToSvgByteArray(byte[] input) {
        ByteArrayOutputStreamSource baos = new ByteArrayOutputStreamSource();
        transcodeWmfToSvg(new TranscoderInput(new ByteArrayInputStream(input)), baos);
        return baos.getByteArray();
    }

}
