package ru.yandex.chemodan.app.docviewer.adapters.freehep;

import java.awt.Dimension;
import java.util.List;
import java.util.Properties;

import org.freehep.graphicsbase.util.export.ExportFileType;
import org.freehep.graphicsio.ImageConstants;
import org.freehep.graphicsio.emf.EMFInputStream;
import org.freehep.graphicsio.emf.EMFPanel;
import org.freehep.graphicsio.emf.EMFRenderer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.function.Function1V;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.OutputStreamSource;
import ru.yandex.misc.thread.ThreadLocalTimeout;

public class FreeHepAdapter {

    private static final Logger logger = LoggerFactory.getLogger(FreeHepAdapter.class);

    private final int maxDimSizeLimit;

    public FreeHepAdapter(int maxDimSizeLimit) {
        this.maxDimSizeLimit = maxDimSizeLimit;
    }

    public void convertEmfToSvg(InputStreamSource inputStreamSource,
            final OutputStreamSource outputStreamSource)
    {
        ThreadLocalTimeout.check();

        final ExportFileType exportFileType;
        {
            final List<?> exportFileTypes = ExportFileType.getExportFileTypes(ImageConstants.SVG);
            if (exportFileTypes == null || exportFileTypes.isEmpty()) {
                throw new AssertionError("SVG library is not available on classpath");
            }
            exportFileType = (ExportFileType) exportFileTypes.get(0);
        }

        logger.info("Converting '{}' into '{}' with FreeHEP EMF-to-SVG converter...",
                inputStreamSource, outputStreamSource);

        withEmfRenderer(inputStreamSource, emfRenderer -> {
            final EMFPanel emfPanel = new EMFPanel();
            emfPanel.setRenderer(emfRenderer);

            Dimension dimension = emfRenderer.getSize();
            logger.info("Emf image dimension: " + dimension);

            if (dimension.getWidth() == -1 || dimension.getHeight() == -1) {
                throw new IllegalArgumentException("Unable to read SVG: dimensions are " + dimension);
            } else if (dimension.getWidth() > maxDimSizeLimit || dimension.getHeight() > maxDimSizeLimit) {
                throw new IllegalArgumentException("Emf image size too big: " + dimension);
            }

            ThreadLocalTimeout.check();

            outputStreamSource.write(outputSvg -> {
                try {
                    exportFileType.exportToFile(outputSvg, emfPanel, emfPanel,
                            new Properties(), "Freehep EMF converter");
                } catch (Exception exc) {
                    throw ExceptionUtils.translate(exc);
                }
            });
        });

    }

    private void withEmfRenderer(final InputStreamSource inputStreamSource,
            final Function1V<EMFRenderer> handler)
    {
        inputStreamSource.readNr(a -> {
            try {
                final EMFInputStream emfInputStream = new EMFInputStream(a,
                        new ExtendedEmfTagSet(EMFInputStream.DEFAULT_VERSION));
                EMFRenderer emfRenderer = new EMFRenderer(emfInputStream);
                handler.apply(emfRenderer);
            } catch (Exception exc) {
                throw ExceptionUtils.translate(exc);
            }
        });

    }
}
